//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: FFLexer.cpp,v 1.16 2004/03/29 22:56:34 delpinux Exp $

#include <StreamCenter.hpp>

#include <Expression.hpp>
#include <RealExpression.hpp>
#include <BooleanExpression.hpp>
#include <Vector3Expression.hpp>
#include <FunctionExpression.hpp>

#include <MeshExpression.hpp>

#include <DomainExpression.hpp>
#include <InsideExpression.hpp>
#include <InsideListExpression.hpp>

#include <UnknownExpression.hpp>
#include <UnknownListExpression.hpp>
#include <BoundaryExpression.hpp>
#include <BoundaryConditionExpression.hpp>
#include <BoundaryConditionListExpression.hpp>

#include <PDEOperatorExpression.hpp>

#include <PDESystemExpression.hpp>
#include <PDEEquationExpression.hpp>

#include <SolverOptionsExpression.hpp>
#include <SubOptionListExpression.hpp>
#include <SubOptionExpression.hpp>

#include <Instruction.hpp>

#include <TestFunctionExpression.hpp>
#include <IntegratedExpression.hpp>
#include <IntegratedOperatorExpression.hpp>

#include <LinearExpression.hpp>
#include <MultiLinearExpression.hpp>
#include <MultiLinearFormExpression.hpp>

#include <VariationalFormulaExpression.hpp>
#include <VariationalProblemExpression.hpp>

#include <VariationalDirichletListExpression.hpp>

#include <OstreamExpressionList.hpp>

#include <parse.ff.h>

#include <cctype>

#include <FFLexer.hpp>

#include <sstream>

#include <VariableList.hpp>
extern VariableList variableList;

#include <ParameterCenter.hpp>

//! Constructs a FFLexer for given std::istream and std::ostream.
FFLexer::FFLexer(std::istream& In,
		 std::ostream& Out)
  : Lexer(In,Out)
{
  __keyWordList["x"] = XX;
  __keyWordList["y"] = YY;
  __keyWordList["z"] = ZZ;
  __keyWordList["double"] = DOUBLE;
  __keyWordList["femfunction"] = FEMFUNCTION;
  __keyWordList["function"] = FUNCTION;
  __keyWordList["mesh"] = MESH;
  __keyWordList["structured"] = STRUCTMESH;
  __keyWordList["surface"] = SURFMESH;
  __keyWordList["tetrahedrize"] = TETRAHEDRIZE;
  __keyWordList["transform"] = TRANSFORM;
  __keyWordList["simplify"] = SIMPLIFY;
  __keyWordList["extract"] = EXTRACT;

 // Those two keywords are equivalent!
  __keyWordList["vector"] = VECTOR;
  __keyWordList["vertex"] = VECTOR;

  __keyWordList["scene"] = SCENE;
  __keyWordList["domain"] = DOMAIN_;

  // statements
  __keyWordList["do"] = DO;
  __keyWordList["while"] = WHILE;
  __keyWordList["for"] = FOR;
  __keyWordList["if"] = IF;
  __keyWordList["else"] = ELSE;

  __keyWordList["true"] = TRUE;
  __keyWordList["false"] = FALSE;
  __keyWordList["not"] = NOT;
  __keyWordList["and"] = AND;
  __keyWordList["or"] = OR;
  __keyWordList["xor"] = XOR;

  __keyWordList["solve"] = SOLVE;

  __keyWordList["div"] = DIV;
  __keyWordList["grad"] = GRAD;
  __keyWordList["dnu"] = DNU;

  __keyWordList["int"] = INT;
  __keyWordList["max"] = MAXIMUM;
  __keyWordList["int1d"] = INT1D;
  __keyWordList["int2d"] = INT2D;
  __keyWordList["int3d"] = INT3D;

  __keyWordList["convect"] = CONVECT;

  __keyWordList["pde"] = PDEQ;

  __keyWordList["dx"] = DX;
  __keyWordList["dy"] = DY;
  __keyWordList["dz"] = DZ;

  __keyWordList["in"] = IN;
  __keyWordList["on"] = ON;
  __keyWordList["by"] = BY;
  __keyWordList["using"] = USING;

  __keyWordList["exit"] = EXIT;

  __keyWordList["inside"] = INSIDE;
  __keyWordList["outside"] = OUTSIDE;

  __keyWordList["cout"] = COUT;

  __keyWordList["read"] = READ;
  __keyWordList["save"] = SAVE;
  __keyWordList["plot"] = PLOT;
  __keyWordList["exec"] = EXEC;

  __keyWordList["pov"] = POV;
  __keyWordList["medit"]  = FILEFORMAT;
  __keyWordList["opendx"] = FILEFORMAT;
  __keyWordList["system"] = SYSTEM;

  __keyWordList["abs"] = ABS;
  __keyWordList["sin"] = SIN;
  __keyWordList["cos"] = COS;
  __keyWordList["tan"] = TAN;
  __keyWordList["asin"] = ASIN;
  __keyWordList["acos"] = ACOS;
  __keyWordList["atan"] = ATAN;
  __keyWordList["sqrt"] = SQRT;

  __keyWordList["one"] = ONE;
  __keyWordList["reference"] = REFERENCE;
  __keyWordList["vertices"] = VERTICES;
  __keyWordList["elements"] = ELEMENTS;

  __keyWordList["exp"] = EXP;
  __keyWordList["log"] = LOG;

  __keyWordList["xmin"] = XMIN;
  __keyWordList["xmax"] = XMAX;
  __keyWordList["ymin"] = YMIN;
  __keyWordList["ymax"] = YMAX;
  __keyWordList["zmin"] = ZMIN;
  __keyWordList["zmax"] = ZMAX;

  __keyWordList["dos"]    = FILETYPE;
  __keyWordList["unix"]   = FILETYPE;
  __keyWordList["mac"]    = FILETYPE;
  __keyWordList["binary"] = FILETYPE;

  __keyWordList["test"] = TESTS;

  //! Get the list of keywords
  ParameterCenter::instance().get(__optionsKeyWords);
}

void FFLexer::switchToContext(const FFLexer::Context context)
{
  switch (context) {
  case (FFLexer::defaultContext): {
    __keyWordList["dx"] = DX;
    __keyWordList["dy"] = DY;
    __keyWordList["dz"] = DZ;
    break;
  }
  case (FFLexer::variationalContext): {
    __keyWordList["dx"] = DXOP;
    __keyWordList["dy"] = DYOP;
    __keyWordList["dz"] = DZOP;
    break;
  }
  default: {
    fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
    std::exit(1);
  }
  }
}

/*! This function returns the token associated to the name "word" if it
  corresponds to a Keyword, else, it returns "-1".
*/
int FFLexer::isKeyword(const std::string& word)
{
  KeyWordList::iterator theKeyWord
    = __keyWordList.find(word.c_str());

  if(theKeyWord != __keyWordList.end()) { // if it was found
    switch (theKeyWord->second) {
    case FILETYPE: {
      fflval.fileType = FileDescriptor::fileType(word.c_str());
      break;
    }
    case FILEFORMAT: {
      fflval.formatType = FileDescriptor::formatType(word.c_str());
      break;
    }
    default:{
      ;
    }
    }

    return theKeyWord->second;
  }
  return -1;			// Means this is not a keyword!
}

/*! This function returns the token associated to the name "word" if it
  corresponds to an Option, else, it returns "-1".
*/
int FFLexer::isOption(const std::string& word)
{
  IdentifierSet::iterator theKeyWord
    = __optionsKeyWords.find(word.c_str());
  if(theKeyWord != __optionsKeyWords.end()) { // if it was found
    return OPTION;
  }
  return -1;			// Means this is not a keyword!
}

/*!
  This function returns the token associated to the name "word" if it
  corresponds to a variable, else, it returns "-1".
*/
int FFLexer::isVariable(const std::string& word)
{
  ReferenceCounting<Variable> variable = variableList.find(word);
  if (variable == 0) {
    return -1; // Not a known variable
  } else {
    // Make the variable point on the right region.
    Variable* v = variable;
    switch ((*variable).type()) {
    case (Variable::real): {
      fflval.realVariable = static_cast<RealVariable*>(v);
      return VARREALID;
    }
    case (Variable::vector3): {
      fflval.vector3Variable = static_cast<Vector3Variable*>(v);
      return VARVECTID;
    }
    case (Variable::function): {
      fflval.functionVariable = static_cast<FunctionVariable*>(v);
      if ((*fflval.functionVariable).isSubscribed()) {
	return VARUNKNOWNID;
      } else {
	switch ((*static_cast<FunctionVariable&>(*v).expression()).type()) {
	case FunctionExpression::fem: {
	  return VARFEMFNCTID;
	}
	default:
	  return VARFNCTID;
	}
      }
    }
    case (Variable::mesh): {
      fflval.meshVariable = static_cast<MeshVariable*>(v);
      return MESHID;
    }
    case (Variable::scene): {
      fflval.sceneVariable = static_cast<SceneVariable*>(v);
      return VARSCENEID;
    }
    case (Variable::domain): {
      fflval.domainVariable = static_cast<DomainVariable*>(v);
      return VARDOMAINID;
    }
    case (Variable::testFuncion): {
      fflval.testFunction = static_cast<TestFunctionVariable*>(v);
      return VARTESTFUNCTIONID;
    }
    default: {
      fferr(2) << '\n' << __FILE__ << ':' << __LINE__
	       << ':' << "Not implemented\n";
      std::exit(1);
    }
    }
  }
  return 0;
}

//! The lexer function.
int FFLexer::yylex()
{
  char readChar = ' ';
  in.unsetf(std::ios::skipws);

  try {

    while ((in)&&(isspace(readChar))) {
      in >> readChar;
      if (readChar=='\n')
	linenumber++;
    }

    char nextChar = in.peek(); // read next Char
    ffout(3) << '.' << std::flush; // shows the parsing state.

    if (in) {

      // Is input a Number?
      if (isdigit(readChar) || ((readChar=='.') && isdigit(nextChar))) {
	in.unget();

	if (in >> fflval.value) {
	  std::stringstream is;
	  is << fflval.value << std::ends;
	  yytext = is.str();

	  return NUM;
	}
	else {
	  fferr(2) << '\n' << __FILE__ << ':' << __LINE__
		   << ':' << "Not implemented\n";
	  std::exit(1);
	}
      }

      if (isalpha(readChar)) {
	std::string is;
	while (isalnum(readChar)) {
	  is += readChar;
	  in >> readChar;
	}
	in.unget();

	yytext = is;
	// is the word a Keyword?
	int TOKEN = isKeyword(yytext);
	if (TOKEN != -1) {
	  return TOKEN;
	}
	// reaching this place means this is not a Keyword!

	// Is the TOKEN an option ?
	TOKEN = isOption(yytext);
	if(TOKEN != -1) {
	  fflval.str = new char[strlen(yytext.c_str())+1];
	  strcpy (fflval.str, yytext.c_str());
	  return TOKEN;
	}
	
	// is the word a Variable?
	TOKEN = isVariable(yytext);
	if (TOKEN != -1) {
	  return TOKEN;
	} else {
	  fflval.aString = new std::string(yytext);
	  return NAME;
	}
	// reaching this place means this is not a Variable!

      }

      // Comments treatment.
      switch(readChar) {
      case '/': {
	if (nextChar == '/') {	// end of line comment
	  while (readChar!='\n')
	    in >> readChar;
	  in.unget();
	  return yylex();	// use recursion
	}

	if (nextChar == '*') {	// multi-line comment
	  bool endOfComment = false;
	  size_t commentLines = 0; // Counts lines within comment

	  in >> readChar;
	  while (!endOfComment) {
	    if(in >> readChar) {
	      switch (readChar) {
	      case '*': {
		if  (in.peek() == '/') {
		  endOfComment = true;
		}
		break;
	      }
	      case '\n': {
		commentLines++;
		break;
	      }
	      }
	    } else {
	      fferr(3) << linenumber <<": opened comment never closed\n";
	      exit(1);
	    }
	  }
	  in >> readChar;

	  linenumber += commentLines;
	  return yylex();
	}
      }
      case '+': {
	if (nextChar == '+') {	// increment operator
	  in >> readChar;
	  return INCR;
	}
	break;
      }

      case '-': {
	if (nextChar == '-') {	// increment operator
	  in >> readChar;
	  yytext = "--";
	  return DECR;
	}
	break;
      }
      case '<':{
	if (nextChar == '=') {	// increment operator
	  in >> readChar;
	  yytext = "<=";
	  return LEQ;
	} else if (nextChar == '<') {	// increment operator
	  in >> readChar;
	  yytext = "<<";
	  return LRAFTER;
	}
	yytext = "<";
	return LT;
      }
      case '>':{
	if (nextChar == '=') {	// increment operator
	  in >> readChar;
	  yytext = ">=";
	  return GEQ;
	} else if (nextChar == '>') {	// increment operator
	  in >> readChar;
	  yytext = ">>";
	  return RRAFTER;
	}
	yytext = ">";
	return GT;
      }
      case '=': {
	if (nextChar == '=') {	// increment operator
	  in >> readChar;
	  yytext = "==";
	  return EQ;
	}
	break;
      }
      case '!': {
	if (nextChar == '=') {	// increment operator
	  in >> readChar;
	  yytext = "!=";
	  return NE;
	}
	return NOT;
      }
      case '|': {
	if (nextChar == '|') {	// increment operator
	  yytext = "||";
	  in >> readChar;
	  return OR;
	}
	break;
      }
      case '&': {
	if (nextChar == '&') {	// increment operator
	  in >> readChar;
	  yytext = "&&";
	  return AND;
	}
	break;
      }

      case ')': {
	if (declarationToken == SOLVE)	// Declaration has been achieved.
	  declarationToken = -1;        // Must remove Type storage!
	return ')';
      }

      case '"': {
	bool endOfString = false;
	size_t stringLines = 0; // Counts lines within the string
	std::stringstream is;

	while (!endOfString) {
	  if(in >> readChar) {
	    switch (readChar) {
	    case '"': {
	      endOfString = true;
	      break;
	    }
	    case '\n': {
	      stringLines++;
	      is << '\n';
	      break;
	    }
	    case '\\': {
	      int peekC = in.peek();
	      switch (peekC) {
	      case 'n': {
		is << '\n';
		in >> readChar;
		break;
	      }
	      case 't': {
		is << '\t';
		in >> readChar;
		break;
	      }
	      case 'v': {
		is << '\v';
		in >> readChar;
		break;
	      }
	      case 'b': {
		is << '\b';
		in >> readChar;
		break;
	      }
	      case 'r': {
		is << '\r';
		in >> readChar;
		break;
	      }
	      case 'f': {
		is << '\f';
		in >> readChar;
		break;
	      }
	      case 'a': {
		is << '\a';
		in >> readChar;
		break;
	      }
	      case '\'': {
		is << '\'';
		in >> readChar;
		break;
	      }
	      case '"': {
		is << '"';
		in >> readChar;
		break;
	      }
	      case '\\': {
		is << '\\';
		in >> readChar;
		break;
	      }
	      default: {
		fferr(0) << linenumber << ": Unknown escape sequence\n";
		exit(1);
	      }
	      }
	      break;
	    }
	    default: {
	      is << readChar;
	    }
	    }
	  } else {
	    fferr(2) << linenumber <<": opened string never closed\n";
	    exit(1);
	  }
	}
	is << std::ends;

	fflval.str = new char[is.str().size()+1];
	strcpy (fflval.str, is.str().c_str());

	linenumber += stringLines;
	//	yytext = fflval.str;
	return STRING;
      }

      }

      // Others characters treatment
      yytext = readChar;
      return readChar;
    } else {
      return EOF;
    }

  } // try

  catch (...) {
    fferr(2) << '\n' << __FILE__ << ':' << __LINE__
	     << ':' << "Not implemented\n";
    std::exit(1);
  }
  return 0;
}

