//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: MeshExpression.cpp,v 1.16 2004/03/15 21:01:20 delpinux Exp $

#include <MeshExpression.hpp>
#include <Variable.hpp>

#include <Mesh.hpp>

#include <MeshOfHexahedra.hpp>
#include <MeshOfTetrahedra.hpp>
#include <Structured3DMesh.hpp>

#include <SurfaceMeshOfQuadrangles.hpp>
#include <SurfaceMeshOfTriangles.hpp>

#include <Domain.hpp>
#include <Scene.hpp>

#include <DomainExpression.hpp>

#include <Union.hpp>

#include <MeshTetrahedrizor.hpp>
#include <MeshSimplifier.hpp>
#include <MeshReader.hpp>
#include <MeshExtractor.hpp>

#include <SurfaceMeshGenerator.hpp>
#include <RealExpression.hpp>

ReferenceCounting<Mesh> MeshExpression::mesh() const
{
  assert(__typeOfMesh != MeshExpression::undefined);
  return __mesh;
}

MeshExpression::MeshExpression(const MeshExpression& e)
  : Expression(e),
    __mesh(e.__mesh),
    __typeOfMesh(e.__typeOfMesh)
{
  ;
}

MeshExpression::MeshExpression(ReferenceCounting<Mesh> mesh,
			       const MeshExpression::TypeOfMesh& t)
  : Expression(Expression::mesh),
    __mesh(mesh),
    __typeOfMesh(t)
{
  ;
}

MeshExpression::~MeshExpression()
{
  ;
}


std::ostream& MeshExpressionStructured::put(std::ostream& os) const
{
  const Structured3DMesh& smesh = static_cast<const Structured3DMesh&>(*__mesh);
  os << "structured mesh of {(" << smesh.shape().shape().nx() << ','
     << smesh.shape().shape().ny() << ','<< smesh.shape().shape().nz()
     << ");" << smesh.shape().a() << ';' << smesh.shape().b() << '}';
  return os;
}

void MeshExpressionStructured::execute()
{
  TinyVector<3> a;
  TinyVector<3> b;
  TinyVector<3,int> n;
  (*__corner1).execute();
  (*__corner2).execute();
  (*__meshSize).execute();
  for (size_t i=0; i<3; ++i) {
    a[i] = (*__corner1).value(i);
    b[i] = (*__corner2).value(i);
    n[i] = static_cast<int>((*__meshSize).value(i));
  }
  ffout(2) << "Building Mesh ... ";
  Structured3DMeshShape S(n[0], n[1], n[2], a, b);
  __mesh = new Structured3DMesh(S);
  ffout(2) << "done\n";
}

MeshExpressionStructured::
MeshExpressionStructured(ReferenceCounting<Vector3Expression> size,
			 ReferenceCounting<Vector3Expression> corner1,
			 ReferenceCounting<Vector3Expression> corner2)
  : MeshExpression(0, MeshExpression::structured),
    __meshSize(size),
    __corner1(corner1),
    __corner2(corner2)
{
  ;
}


MeshExpressionStructured::
MeshExpressionStructured(const MeshExpressionStructured& m)
  : MeshExpression(m),
    __meshSize(m.__meshSize),
    __corner1(m.__corner1),
    __corner2(m.__corner2)
{
  ;
}

MeshExpressionStructured::
~MeshExpressionStructured()
{
  ;
}




std::ostream& MeshExpressionSurface::put(std::ostream& os) const
{
  fferr(2) << __FILE__ << ':' << __LINE__  << ": Not Implemented\n";
  std::exit(1);
  return os;
}

void MeshExpressionSurface::execute()
{
  (*__domain).execute();
  (*__smesh).execute();

  __mesh = new SurfaceMeshOfTriangles();

  Structured3DMesh& SMesh
    = dynamic_cast<Structured3DMesh&>(*(*__smesh).mesh());
  SurfaceMeshOfTriangles& surfaceMesh
    = dynamic_cast<SurfaceMeshOfTriangles&>(*__mesh);

  Domain& omega = (*(*__domain).domain());

  if(omega.isR3()) {
    fferr(0) << "You are trying to mesh a domain which is equal to R3!\n";
    std::exit(1);
  }

  SurfaceMeshGenerator S;
  S.generateSurfacicMesh(omega,
			 SMesh,
			 surfaceMesh);
}

MeshExpressionSurface::
MeshExpressionSurface(ReferenceCounting<DomainExpression> domain,
		      ReferenceCounting<MeshExpression> smesh)
  : MeshExpression(0, MeshExpression::surface),
    __domain(domain),
    __smesh(smesh)
{
  ;
}


MeshExpressionSurface::
MeshExpressionSurface(const MeshExpressionSurface& m)
  : MeshExpression(m),
    __domain(m.__domain),
    __smesh(m.__smesh)
{
  ;
}

MeshExpressionSurface::
~MeshExpressionSurface()
{
  ;
}


void MeshExpressionVariable::execute()
{
  __mesh = (*(*__meshVariable).expression()).mesh();
}

MeshExpressionVariable::MeshExpressionVariable(ReferenceCounting<MeshVariable> r)
  : MeshExpression(0,
		   MeshExpression::variable),
    __meshVariable(r)
{
  ;
}

MeshExpressionVariable::MeshExpressionVariable(const MeshExpressionVariable& e)
  : MeshExpression(e),
    __meshVariable(e.__meshVariable)
{
  ;
}

MeshExpressionVariable::~MeshExpressionVariable()
{
  ;
}

std::ostream&
MeshExpressionRead::put(std::ostream& os) const
{
  os << "read(medit," << (*__filename).value() << ')';
  return os;
}

void MeshExpressionRead::execute()
{
  (*__filename).execute();
  std::string filename = (*__filename).value();

  std::ifstream is(filename.c_str());
  if (not(is)) {
    fferr(0) << "error: cannot open file '" << filename << "'\n";
    std::exit(1);
  }
  MeshReader M(is, filename);

  __mesh = M.mesh();
}

MeshExpressionRead::
MeshExpressionRead(ReferenceCounting<FileDescriptor> descriptor,
		   ReferenceCounting<StringExpression> filename)
  : MeshExpression(0, MeshExpression::read),
    __fileDescriptor(descriptor),
    __filename(filename)
{
  ;
}

MeshExpressionRead::
MeshExpressionRead(const MeshExpressionRead& m)
  : MeshExpression(m),
    __fileDescriptor(m.__fileDescriptor),
    __filename(m.__filename)
{
  ;
}

MeshExpressionRead::~MeshExpressionRead()
{
  ;
}

std::ostream&
MeshExpressionSimplify::
put(std::ostream& os) const
{
  os << "simplify("
     << (*__originalMesh) << ')';
  return os;
}

void
MeshExpressionSimplify::
execute()
{
  (*__originalMesh).execute();
  MeshSimplifier M((*__originalMesh).mesh());
  __mesh = M.mesh();
}

MeshExpressionSimplify::
MeshExpressionSimplify(ReferenceCounting<MeshExpression> me)
  : MeshExpression(0, MeshExpression::simplify),
    __originalMesh(me)
{
  ;
}

MeshExpressionSimplify::
MeshExpressionSimplify(const MeshExpressionSimplify& e)
  : MeshExpression(e),
    __originalMesh(e.__originalMesh)
{
  ;
}

MeshExpressionSimplify::
~MeshExpressionSimplify()
{
  ;
}


std::ostream&
MeshExpressionExtract::
put(std::ostream& os) const
{
  os << "extract(" << (*__originalMesh) << ','
     << (*__referenceToExtract) << ')';
  return os;
}

template <typename ExtractedMeshType>
void MeshExpressionExtract::__extract()
{
   MeshExtractor<ExtractedMeshType>
     extractor(static_cast<const ExtractedMeshType&>(*(*__originalMesh).mesh()));

   const int ref = static_cast<int>((*__referenceToExtract).realValue());
   if (ref < 0) {
     fferr(0) << "Element references have to be positives\n";
     std::exit(1);
   }
   std::set<size_t> refSet;
   refSet.insert(static_cast<size_t>(ref));
   __mesh = extractor(refSet);
}

void
MeshExpressionExtract::
execute()
{
  (*__originalMesh).execute();
  (*__referenceToExtract).execute();

  switch ((*(*__originalMesh).mesh()).type()) {
//   case Mesh::cartesianHexahedraMesh: {
//     this->__extract<Structured3DMesh>();
//     break;
//   }
//   case Mesh::hexahedraMesh: {
//     this->__extract<MeshOfHexahedra>();
//     break;
//   }
  case Mesh::tetrahedraMesh: {
    this->__extract<MeshOfTetrahedra>();
    break;
  }
  case Mesh::surfaceMeshQuadrangles: {
    this->__extract<SurfaceMeshOfQuadrangles>();
    break;
  }
  case Mesh::surfaceMeshTriangles: {
    this->__extract<SurfaceMeshOfTriangles>();
    break;
  }
  default: {
    fferr(0) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
    std::exit(1);
  }
  }
}

MeshExpressionExtract::
MeshExpressionExtract(ReferenceCounting<MeshExpression> me,
		      ReferenceCounting<RealExpression> re)
  : MeshExpression(0, MeshExpression::extract),
    __originalMesh(me),
    __referenceToExtract(re)
{
  ;
}

MeshExpressionExtract::
MeshExpressionExtract(const MeshExpressionExtract& e)
  : MeshExpression(e),
    __originalMesh(e.__originalMesh),
    __referenceToExtract(e.__referenceToExtract)
{
  ;
}

MeshExpressionExtract::
~MeshExpressionExtract()
{
  ;
}



void
MeshExpressionTetrahedrize::execute()
{
  (*__inputMesh).execute();
  ffout(3) << "Converting to tetrahedra "
	   << (*__inputMesh);
  MeshTetrahedrizor mt((*__inputMesh).mesh());

  mt.run();

  __mesh = mt.mesh();
  ffout(3) << "\ndone\n";
}

MeshExpressionTetrahedrize::
MeshExpressionTetrahedrize(ReferenceCounting<MeshExpression> m)
  : MeshExpression(0,MeshExpression::tetrahedrize),
    __inputMesh(m)
{
  ;
}

MeshExpressionTetrahedrize::
MeshExpressionTetrahedrize(const MeshExpressionTetrahedrize& m)
  : MeshExpression(m),
    __inputMesh(m.__inputMesh)
{
  ;
}

MeshExpressionTetrahedrize::
~MeshExpressionTetrahedrize()
{
  ;
}

template <typename ElementGeometry,
	  typename MeshType>
ElementGeometry MeshExpressionTransform::
__getElement(const MeshType& M,
	     VerticesSet& V,
	     size_t numberOfCell)
{
  fferr(0) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
  std::exit(1);
}

template <>
Hexahedron MeshExpressionTransform::
__getElement<Hexahedron>(const Structured3DMesh& m,
			 VerticesSet& V,
			 size_t i)
{
  const CartesianHexahedron& H = m.cell(i);
  const size_t i0 = m.vertexNumber(H(0));
  const size_t i1 = m.vertexNumber(H(1));
  const size_t i2 = m.vertexNumber(H(2));
  const size_t i3 = m.vertexNumber(H(3));
  const size_t i4 = m.vertexNumber(H(4));
  const size_t i5 = m.vertexNumber(H(5));
  const size_t i6 = m.vertexNumber(H(6));
  const size_t i7 = m.vertexNumber(H(7));

  return Hexahedron(V[i0],V[i1],V[i2],V[i3],
		    V[i4],V[i5],V[i6],V[i7]);
}

template <>
Tetrahedron MeshExpressionTransform::
__getElement<Tetrahedron>(const MeshOfTetrahedra& m,
			 VerticesSet& V,
			 size_t i)
{
  const Tetrahedron& T = m.cell(i);
  const size_t i0 = m.vertexNumber(T(0));
  const size_t i1 = m.vertexNumber(T(1));
  const size_t i2 = m.vertexNumber(T(2));
  const size_t i3 = m.vertexNumber(T(3));

  return Tetrahedron(V[i0],V[i1],V[i2],V[i3]);
}

template <typename MeshType>
void MeshExpressionTransform::__transform(const Mesh& M)
{
  fferr(0) << __FILE__ << ':' << __LINE__
	   << ": THIS CODE IS NOT AT THE RIGHT PLACE\n";
  std::exit(1);
  const MeshType& m = static_cast<const MeshType&>(M);

  typedef typename MeshType::Transformed NewMeshType;
  typedef typename NewMeshType::ElementGeometry ElementGeometry;

  ReferenceCounting<VerticesSet> v = new VerticesSet(m.numberOfVertices());

  VerticesSet& vertices = *v;
  FunctionExpression& f1 = (*__transformationField).f1();
  FunctionExpression& f2 = (*__transformationField).f2();
  FunctionExpression& f3 = (*__transformationField).f3();

  for (size_t i=0; i<m.numberOfVertices(); ++i) {
    const Vertex& X = m.vertex(i);
    vertices[i][0] = f1.value(X[0],X[1],X[2]);
    vertices[i][1] = f2.value(X[0],X[1],X[2]);
    vertices[i][2] = f3.value(X[0],X[1],X[2]);
    vertices[i].reference() = X.reference();
  }

  ReferenceCounting<Vector<ElementGeometry> > elements
    = new Vector<ElementGeometry>(m.numberOfCells());


  for (size_t i=0; i<m.numberOfCells(); ++i) {
    (*elements)[i] = __getElement<ElementGeometry>(m, vertices, i);
  }

  __mesh = new NewMeshType(v, elements);
}


void
MeshExpressionTransform::execute()
{
  (*__inputMesh).execute();
  (*__transformationField).execute();

  const Mesh& M = (*(*__inputMesh).mesh());
  switch (M.type()) {
  case Mesh::cartesianHexahedraMesh: {
    this->__transform<Structured3DMesh>(M);
    break;
  }
  case Mesh::tetrahedraMesh: {
    this->__transform<MeshOfTetrahedra>(M);
    break;
  }
  default: {
    fferr(0) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
    std::exit(1);
  }
  }
}

MeshExpressionTransform::
MeshExpressionTransform(ReferenceCounting<MeshExpression> m,
			ReferenceCounting<FieldExpression> f)
  : MeshExpression(0,MeshExpression::transform),
    __inputMesh(m),
    __transformationField(f)
{
  ;
}

MeshExpressionTransform::
MeshExpressionTransform(const MeshExpressionTransform& m)
  : MeshExpression(m),
    __inputMesh(m.__inputMesh),
    __transformationField(m.__transformationField)
{
  ;
}

MeshExpressionTransform::
~MeshExpressionTransform()
{
  ;
}


MeshExpressionUndefined::MeshExpressionUndefined()
  : MeshExpression(0, MeshExpression::undefined)
{
  ;
}

MeshExpressionUndefined
::MeshExpressionUndefined(const MeshExpressionUndefined& m)
  : MeshExpression(m)
{
  ;
}

MeshExpressionUndefined
::~MeshExpressionUndefined()
{
  ;
}

