//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: VectorialPDEOperator.hpp,v 1.2 2003/09/20 21:31:09 delpinux Exp $

// This class describes a vectorial PDE operator.
// It means a PDE operator that can be applied to a vectorial function.


#ifndef VECTORIALPDEOPERATOR_HPP
#define VECTORIALPDEOPERATOR_HPP

#include <vector>
#include <PDEOperatorSum.hpp>

/*!
  \class VectorialPDEOperator

  This class descibes a vectorial PDE operator. This means that 

  \author Stphane Del Pino
*/
class VectorialPDEOperator
{
private:
  std::vector<ReferenceCounting<PDEOperatorSum> > __vectorPDEOperator;

public:
  const size_t size() const
  {
    return __vectorPDEOperator.size();
  }

  //! access to the ith Component of the vectorial PDE operator
  PDEOperatorSum& operator [] (const size_t i)
  {
    assert(i<__vectorPDEOperator.size());
    return (*__vectorPDEOperator[i]);
  }

  //! Read-only access to the ith Component of the vectorial PDE operator
  const PDEOperatorSum& operator [] (const size_t i) const
  {
    assert(i<__vectorPDEOperator.size());
    return (*__vectorPDEOperator[i]);
  }

  friend std::ostream& operator << (std::ostream& os,
				    const VectorialPDEOperator& vectPDEOp)
  {
    if (vectPDEOp.__vectorPDEOperator.size() > 0) {
      for(size_t i=0; i<vectPDEOp.__vectorPDEOperator.size(); ++i) {
	if (i!=0)
	  os << " + ";
	os << '(' << *(vectPDEOp.__vectorPDEOperator[i]) << ")(u_" << i << ')';
      }
    } else {
      os << 0;
    }
    return os;
  }

  ReferenceCounting<VectorialPDEOperator> operator * (ConstReferenceCounting<UserFunction> u) const
  {
    VectorialPDEOperator* newVectorialPDEOperator = new VectorialPDEOperator(this->size());

    for (size_t i=0; i<__vectorPDEOperator.size(); ++i)
      (*newVectorialPDEOperator).__vectorPDEOperator[i] = (*__vectorPDEOperator[i]) * u;

    return newVectorialPDEOperator;
  }

  const VectorialPDEOperator& operator += (const VectorialPDEOperator& VPDEOp)
  {
    assert(__vectorPDEOperator.size() == VPDEOp.__vectorPDEOperator.size());

    for (size_t i=0; i<__vectorPDEOperator.size(); ++i)
      for (size_t j=0; j<(*(VPDEOp.__vectorPDEOperator[i])).nbOperators(); ++j)
	(*__vectorPDEOperator[i]).add((*(VPDEOp.__vectorPDEOperator[i]))[j]);

    return *this;
  }

  const VectorialPDEOperator& operator -= (const VectorialPDEOperator& VPDEOp)
  {
    assert(__vectorPDEOperator.size() == VPDEOp.__vectorPDEOperator.size());

    for (size_t i=0; i<__vectorPDEOperator.size(); ++i)
      for (size_t j=0; j<(*VPDEOp.__vectorPDEOperator[i]).nbOperators(); ++j)
	(*__vectorPDEOperator[i]).add((*VPDEOp.__vectorPDEOperator[i])[j]);

    return *this;
  }

  VectorialPDEOperator(const size_t& dimension)
    : __vectorPDEOperator(dimension)
  {
    for(size_t i=0; i<dimension; ++i)
      __vectorPDEOperator[i] = new PDEOperatorSum();
  }
  
  VectorialPDEOperator()
  {
    ;
  }

  ~VectorialPDEOperator()
  {
    ;
  }
};

#endif // VECTORIALPDEOPERATOR_HPP
