/* 

                          Firewall Builder

                 Copyright (C) 2004 NetCitadel, LLC

  Author:  Vadim Kurland vadim@fwbuilder.org

  $Id: RCS.cpp,v 1.36 2004/08/05 05:32:44 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"

// need this for FS_SEPARATOR
#include "fwbuilder/libfwbuilder-config.h"
#include "fwbuilder/Tools.h"

#include "FWWindow.h"

#include <RCS.h>
#include <qdir.h>
#include <qregexp.h>
#include <qmessagebox.h>

#include <stdlib.h>

#if defined(_WIN32)
#  include <stdio.h>
#  include <sys/timeb.h>
#  include <sys/stat.h>
#  include <fcntl.h>
#  include <time.h>
#else
#  include <unistd.h>
#  include <string.h>
#  include <errno.h>
#endif

#include <iostream>

using namespace std;
using namespace libfwbuilder;

QString     RCS::rcs_file_name     = "";
QString     RCS::rlog_file_name    = "";
QString     RCS::rcsdiff_file_name = "";
QString     RCS::ci_file_name      = "";
QString     RCS::co_file_name      = "";

RCSEnvFix*  RCS::rcsenvfix         = NULL;

/***********************************************************************
 *
 * class Revision
 *
 ***********************************************************************/
Revision::Revision()
{
}

Revision::Revision(const QString &file, const QString &r)
{
    filename = file;
    rev      = r;
}

Revision::Revision(const Revision &r)
{
    filename   = r.filename ;
    rev        = r.rev      ;
    date       = r.date     ;
    author     = r.author   ;
    locked_by  = r.locked_by;
    log        = r.log      ;
}

void Revision::operator=(const Revision &r)
{
    filename   = r.filename ;
    rev        = r.rev      ;
    date       = r.date     ;
    author     = r.author   ;
    locked_by  = r.locked_by;
    log        = r.log      ;
}

bool Revision::operator<(const Revision &r)
{
    for(int i=1; ; i++)
    {
        QString v1=  rev.section(".",i,i);
        QString v2=r.rev.section(".",i,i);
        if (v1=="" && v2=="") return false;
        if (v1==v2) continue;
        if (v1=="" && v2!="") return true;
        if (v1!="" && v2=="") return false;
        if (v1.toInt()>v2.toInt()) return false;
        if (v1.toInt()<v2.toInt()) return true;
        i++;
    }
    return true;
}

bool Revision::operator==(const Revision &r)
{
    return rev==r.rev;
}

bool Revision::operator!=(const Revision &r)
{
    return rev!=r.rev;
}

/***********************************************************************
 *
 * class RCSEnvFix
 *
 ***********************************************************************/
RCSEnvFix::RCSEnvFix()
{
#ifdef _WIN32

/* need all this crap because Windows does not set environment
 * variable TZ by default, but rcs absolutely requires it
 */
    struct _timeb timebuffer;

    _tzset();
    _ftime( &timebuffer );
    int tzoffset=timebuffer.timezone;

    QString tz;
    if (tzoffset<0) tz=QString("TZ=GMT+%1").arg(tzoffset/60);
    else            tz=QString("TZ=GMT-%1").arg(tzoffset/60);

    env.push_back(tz);

/*
 * NB: need to prepend installation directory in front of PATH on
 * windows, otherwise ci fails when GUI is launched by windows
 * explorer through file extension association. When the program is
 * launched from menu "Start", its working directory is the dir. where
 * it is installed. Since windows implies a '.' in front of PATH,
 * everything works. When the program is started with some other
 * directory as current dir, RCS tools fail without any error message.
 */
    env.push_back( QString("PATH=")+appRootDir+";"+getenv("PATH") );

#endif

/* also need to set env variable USER for rcs tools, but if the user name
 * contains spaces, replace them with underscores (like "John Smith")
 */
    QString uname=getUserName();

    env.push_back( QString("USER=")+uname);
    env.push_back( QString("LOGNAME=")+uname);
}

QStringList* RCSEnvFix::getEnv()
{
    if (env.empty()) return NULL;
    return &env;
}

/***********************************************************************
 *
 * class RCS
 *
 ***********************************************************************/
RCS::RCS(const QString &file)
{
    if (rcsenvfix==NULL) rcsenvfix = new RCSEnvFix();

    if (rcs_file_name=="")
    {
#ifdef _WIN32
        rcs_file_name     = appRootDir+FS_SEPARATOR+RCS_FILE_NAME      ;
        rlog_file_name    = appRootDir+FS_SEPARATOR+RLOG_FILE_NAME     ;
        rcsdiff_file_name = appRootDir+FS_SEPARATOR+RCSDIFF_FILE_NAME  ;
        ci_file_name      = appRootDir+FS_SEPARATOR+CI_FILE_NAME       ;
        co_file_name      = appRootDir+FS_SEPARATOR+CO_FILE_NAME       ;
#else
        rcs_file_name     = RCS_FILE_NAME      ;
        rlog_file_name    = RLOG_FILE_NAME     ;
        rcsdiff_file_name = RCSDIFF_FILE_NAME  ;
        ci_file_name      = CI_FILE_NAME       ;
        co_file_name      = CO_FILE_NAME       ;
#endif
    }

    filename      = file;
    checked_out   = false;
    locked        = false;
    inrcs         = false;
    tracking_file = false;
    ro            = false;
    temp          = false;

    proc = new QProcess();
    proc->setCommunication( QProcess::Stdout|QProcess::Stderr);

    connect(proc, SIGNAL(readyReadStdout()), this,  SLOT(readFromStdout() ) );
    connect(proc, SIGNAL(readyReadStderr()), this,  SLOT(readFromStderr() ) );


    try
    {
        QString  rcspath=filename.left( filename.findRev("/") );
        QDir     rcsdir;
        rcsdir.cd(rcspath);

        QStringList rcslog=rlog();

        QStringList::iterator i;

        for (i=rcslog.begin(); i!=rcslog.end(); ++i)
        {
            if ( (*i).find("head: ")==0)
            {
                head=(*i).section(QRegExp("\\s+"),1,1);
                continue;
            }

            if ( (*i).find("==========")==0 ) break;
            if ( (*i).find("----------")==0 )
            {
                Revision r(filename);

                ++i;
                r.rev       = (*i).section(QRegExp("\\s+"),1,1);
                QString lb  = (*i).section(QRegExp("[\\s;]+"),4,4);
                if (lb!="")
                {       
                    r.locked_by = lb;
                    locked      = true;
                    locked_by   = lb;
                    locked_rev  = r.rev;
                }
                ++i;
                r.date      = (*i).section(QRegExp("[\\s;]+"),1,2);
                r.author    = QString("    ")+(*i).section(QRegExp("[\\s;]+"),4,4);
                ++i;
                if ( (*i).find("branches: ")==0 ) ++i;
                r.log="";
                do {
                    r.log= r.log + *i;
                    ++i;
                } while ( i!=rcslog.end() &&
                          (*i).find("----------")!=0 &&
                          (*i).find("==========")!=0 );
                --i;

                revisions.push_back(r);
            }
        }
//        qBubbleSort( revisions.begin() , revisions.end() );

        inrcs         = true;
        tracking_file = true;
        selectedRev   = head;
    }
    catch (FWException &ex)
    {
        inrcs         = false;
        tracking_file = true;
    }
}

RCS::~RCS()
{
    delete proc;
}

QStringList* RCS::getEnv() { return rcsenvfix->getEnv(); }

void RCS::readFromStdout()
{
    stdoutBuffer=stdoutBuffer + QString(proc->readStdout());
}

void RCS::readFromStderr()
{
    stderrBuffer=stderrBuffer + QString(proc->readStderr());
}

/*********************************************************************
 *  trivial RCS integration
 */

void RCS::abandon()
{
/* check out head revision and unlock it */
    proc->clearArguments();

    proc->addArgument( co_file_name );
    proc->addArgument( "-q" );
    proc->addArgument( "-f" );
    proc->addArgument( QString("-u") );
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (fwbdebug) qDebug("starting co with environment '%s'",
                         rcsenvfix->getEnv()->join("\n").ascii());

    if (proc->start( rcsenvfix->getEnv() ) )
    {
        while (proc->isRunning()) ; // cxx_sleep(1);
        if (proc->normalExit() && proc->exitStatus()==0)
        {
            checked_out = false;
            locked      = false;
            selectedRev = head;
            return;
        }
    }
/* error. */

    selectedRev = "";

    checked_out=false;
    
    QString err = tr("Error checking file out: %1").arg(stderrBuffer);
    QMessageBox::critical(mw, "Firewall Builder", err, tr("&Continue") );

    throw(FWException(err.latin1()));
}

/**
 *  initial RCS checkin
 */
void RCS::add() throw(libfwbuilder::FWException)
{
    proc->clearArguments();

    int      i=filename.findRev("/");
    QString  rcspath=filename.left(i);
    QDir     rcsdir;
    rcsdir.cd(rcspath);

    if (!rcsdir.exists("RCS")) rcsdir.mkdir("RCS");

    proc->addArgument( rcs_file_name );
    proc->addArgument( "-q" );
    proc->addArgument( "-i" );
    proc->addArgument( "-kb" );
    proc->addArgument( "-t-\"Initial checkin\"" );
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (proc->start( rcsenvfix->getEnv() ) )
    {
        while (proc->isRunning()) ; // cxx_sleep(1);
        if (proc->normalExit() && proc->exitStatus()==0)
        {
            proc->clearArguments();
            proc->addArgument( ci_file_name );
            proc->addArgument( "-q" );
            proc->addArgument( "-u" );
            proc->addArgument( filename );

            stdoutBuffer="";
            stderrBuffer="";

            if (proc->start( rcsenvfix->getEnv() ) )
            {
                while (proc->isRunning()) ; // cxx_sleep(1);
                if (proc->normalExit() && proc->exitStatus()==0)
                {
                    inrcs       = true;
                    selectedRev = "1.1";
                    head        = "1.1";
                    return;
                }
            }
        }
    }
    QByteArray outp = proc->readStdout();
    QString msg=QObject::tr("Fatal error during initial RCS checkin of file %1 :\n %2\nExit status %3")
	.arg(filename).arg(outp.data()).arg(proc->exitStatus());
    throw(FWException( msg.latin1()  ));
}

bool RCS::isInRCS()
{
    if (tracking_file) return inrcs;

    proc->clearArguments();

    proc->addArgument( rlog_file_name );
    proc->addArgument( "-zLT" );
    proc->addArgument( "-R" );
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (!proc->start( rcsenvfix->getEnv() ) ) throw(FWException("Fatal error running rlog "));

    while (proc->isRunning()) ; // cxx_sleep(1);

    if (proc->normalExit() && proc->exitStatus()==1)
    {
/* exist status '1' means the file is not in RCS */
        inrcs=false;
        return false;
    }
    inrcs=true;
    return true;
}

bool RCS::co(bool force) throw(libfwbuilder::FWException)
{
    return co(selectedRev,force);
}

/**
 *  RCS checkout
 *
 * possible situations:
 *
 * 1. file is not in RCS - do nothing, return false
 *
 * 2. need to open file read-only
 *
 *   2.1 requested revision is emty or the head: no need to
 *   checkout, just return true
 *
 *   2.2 need to open read-only, older revision: do checkout of that
 *   revision into temporary file without locking, change file name,
 *   set flag 'temp'
 *
 * 3. need to open read-write, but file is locked
 *
 *   3.1 file is locked by the same user: offer user a choice
 *   open read-only or continue editing or cancel
 *
 *   3.2 file is locked by another user: offer a choice open read-only
 *   or cancel
 *
 * 4. need to open read-write, any revision: do normal checkout and
 * lock
 *
 */
bool RCS::co(const QString &rev,bool force) throw(libfwbuilder::FWException)
{
/* first check if filename is already in RCS */

    if (!isInRCS()) return false;

    if (ro)
    {
        if (rev==head || rev=="") return true;

/* check out requested revision to stdout 
 *
 * TODO: right now it loads the whole file into memory, then writes it
 * to the temp file. It should be more efficient to read and write in
 * chunks.
 *
 */
        proc->clearArguments();

        proc->addArgument( co_file_name );
        proc->addArgument( QString("-q") );
        proc->addArgument( QString("-kb") );
        proc->addArgument( QString("-p")+rev );
        proc->addArgument( filename );

        stdoutBuffer="";
        stderrBuffer="";

        if (fwbdebug) qDebug("starting co with environment '%s'",
                             rcsenvfix->getEnv()->join("\n").ascii());

        if (proc->start( rcsenvfix->getEnv() ) )
        {
            while (proc->isRunning())
            {
                QByteArray ba = proc->readStdout();
                if (ba.size()!=0)  stdoutBuffer=stdoutBuffer + QString(ba);
//                cerr << "read " << ba.size() << " bytes from stdout" << endl;
//                cxx_sleep(1);
            }
            if (proc->normalExit() && proc->exitStatus()==0)
            {

#ifdef _WIN32
                char tname[1024];
                strncpy(tname, filename.left(filename.findRev("/")+1).latin1(),sizeof(tname)-20);
                strcat(tname,"tmpXXXXXX");
                _mktemp(tname);
                int fd = _open(tname, _O_RDWR|_O_CREAT|_O_EXCL|_O_BINARY , _S_IREAD|_S_IWRITE );
#else
                char tname[PATH_MAX];
                strncpy(tname, filename.latin1(), sizeof(tname)-20 );
                strcat(tname,"_temp_XXXXXX");
                int fd = mkstemp(tname);
#endif 
		if (fd<0)
		{
                    QString err = tr("Error creating temporary file ")+tname+QString(" :\n")+strerror(errno);
                    QMessageBox::critical(mw, "Firewall Builder", err, tr("&Continue") );
                    throw(FWException(err.latin1()));
		}
#ifdef _WIN32
                if (_write(fd,stdoutBuffer.latin1(),stdoutBuffer.length() )<0)
		{
		    _close(fd);
#else
                if ( write(fd,stdoutBuffer.latin1(),stdoutBuffer.length() )<0)
		{
		    close(fd);
#endif
                    QString err = tr("Error writing to temporary file ")+tname+QString(" :\n")+strerror(errno);
                    QMessageBox::critical(mw, "Firewall Builder", err, tr("&Continue") );
                    throw(FWException(err.latin1()));
                }
                close(fd);

                filename    = tname;
                temp        = true;
                checked_out = false;
                locked      = false;
                selectedRev = rev;
                return true;
            }
        }

        selectedRev = head;

        QString err = tr("Error checking file out: %1").arg(stderrBuffer);
        QMessageBox::critical(mw, "Firewall Builder", err, tr("&Continue") );
        throw(FWException(err.latin1()));

    } else
    {
        QString me=getUserName();
        if (locked)
        {
/* the file is already locked, can not just check it out like that */

            if (me!=locked_by)
            {
                switch (QMessageBox::warning(
                            mw,"Firewall Builder", 
                            tr("File is opened and locked by %1.\nYou can only open it read-only.")
                            .arg(locked_by),
                            "Open &read-only", "&Cancel", QString::null,
                            0, 1 ) )
                {
                case 0:  ro=true;   return false;
                case 1:  throw(FWException("cancel opening file"));  break;
                }
            }

            if (force) goto checkout;

            switch ( QMessageBox::warning(mw, "Firewall Builder",
                                          tr("Revision %1 of this file has been checked out and locked by you earlier.\n\
The file may be opened in another copy of Firewall Builder or was left opened\n\
after the program crashed.").arg(locked_rev),
                                           tr("Open &read-only"), tr("&Open and continue editing"), tr("&Cancel"),
                                          0, 2 ) )
            {
            case 0:  ro=true;  return false;
            case 1:
/* continue working with the file */
                checked_out = true;
                locked      = true;
                selectedRev = locked_rev;
                return true;
            case 2:  throw(FWException("cancel opening file"));  break;
            }
        }

/* if the user wanted specific revision and it should be opened
 * read-only, we need to check it out into a temporary file without
 * locking
 */

 checkout:

/* check out and lock */
        proc->clearArguments();

        proc->addArgument( co_file_name );
        proc->addArgument( "-q" );
        if (force)  proc->addArgument( "-f" );
        proc->addArgument( QString("-l")+rev );
        proc->addArgument( filename );

        stdoutBuffer="";
        stderrBuffer="";

        if (fwbdebug) qDebug("starting co with environment '%s'",
                             rcsenvfix->getEnv()->join("\n").ascii());

        if (proc->start( rcsenvfix->getEnv() ) )
        {
            while (proc->isRunning()) ; // cxx_sleep(1);
            if (proc->normalExit() && proc->exitStatus()==0)
            {
                checked_out = true;
                locked      = true;
                selectedRev = rev;
                return true;
            }
        }
/* error. */

        selectedRev = head;

        QString err = tr("Error checking file out: %1").arg(stderrBuffer);
        QMessageBox::critical(mw, "Firewall Builder", err, tr("&Continue") );

        throw(FWException(err.latin1()));
    }
    return false;
}


bool RCS::ci( const QString &logmsg,
              bool unlock) throw(libfwbuilder::FWException)
{
/* first check if filename is already in RCS */
    if (!isInRCS()) return false;

    proc->clearArguments();

    proc->addArgument( ci_file_name );
    proc->addArgument( "-q" );
    if (unlock) proc->addArgument( "-u" );
    else        proc->addArgument( "-l" );
    if (logmsg=="") proc->addArgument( QString("-m.") );
    else            proc->addArgument( QString("-m")+logmsg );
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (fwbdebug) qDebug("starting ci with environment '%s'",
                         rcsenvfix->getEnv()->join("\n").ascii());

    if (proc->start( rcsenvfix->getEnv() ) )
    {
        if (fwbdebug) qDebug("ci started");

        while (proc->isRunning()) ; // cxx_sleep(1);

        if (fwbdebug) qDebug("ci exited");

        if (proc->normalExit() && proc->exitStatus()==0)
        {
            if (fwbdebug) qDebug("ci exited normally");
            if (unlock)
            {
                checked_out = false;
                locked      = false;
            }
            return true;
        }
    }
    if (fwbdebug) qDebug("Checkin error: file=%s error=%s",
                         filename.latin1(),stderrBuffer.latin1());

    throw( FWException( (stdoutBuffer+"\n"+
                         stderrBuffer+"\n"+
                         proc->arguments().join(" ")+"\n"+
                         rcsenvfix->getEnv()->join("\n")).ascii() ) );

    return false;
}

QStringList RCS::rlog() throw(libfwbuilder::FWException)
{
    proc->clearArguments();

    proc->addArgument( rlog_file_name );
    proc->addArgument( "-zLT" );
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (!proc->start( rcsenvfix->getEnv() ) )
        throw(FWException("Fatal error running rlog "));

    while (proc->isRunning()) 
    {
	QByteArray ba = proc->readStdout();
	if (ba.size()!=0)  stdoutBuffer=stdoutBuffer + QString(ba);
    }

    if (proc->normalExit() && proc->exitStatus()==0)
        return QStringList::split("\n",stdoutBuffer );
    QString msg=QObject::tr("Fatal error running rlog for %1").arg(filename);
    throw( FWException( msg.latin1() ) );
}

QStringList RCS::rcsdiff(const QString &rev) throw(libfwbuilder::FWException)
{
    isDiff();
    return QStringList::split("\n",stdoutBuffer );
}

bool        RCS::isDiff(const QString &rev) throw(libfwbuilder::FWException)
{
    proc->clearArguments();

    proc->addArgument( rcsdiff_file_name );
    proc->addArgument( "-q" );
    if (rev!="") proc->addArgument( QString("-r")+rev );
    else
    {
        if (selectedRev!="") proc->addArgument( QString("-r")+selectedRev );
    }
    proc->addArgument( filename );

    stdoutBuffer="";
    stderrBuffer="";

    if (proc->start( rcsenvfix->getEnv() ) ) 
    {
        while (proc->isRunning())
        {
            QByteArray ba = proc->readStdout();
            if (ba.size()!=0)  stdoutBuffer=stdoutBuffer + QString(ba);
        }
    } else
        throw(FWException("Fatal error running rcsdiff "));

//    while (proc->isRunning()) ; // cxx_sleep(1);

    if (proc->normalExit()) return (proc->exitStatus()!=0);
    QString msg=QObject::tr("Fatal error running rcsdiff for file %1").arg(filename);
    throw( FWException( msg.latin1() ) );
}

QString RCS::getHead()
{
    if (isInRCS()) return head;
    return "";
}

QString RCS::getSelectedRev()
{
    if (isInRCS()) return selectedRev;
    return "";
}

