# encoding: utf8

import os
from binascii import unhexlify

from nose.tools import eq_, assert_raises

import mapnik

from .utilities import execution_path, run_all

try:
    import json
except ImportError:
    import simplejson as json


def setup():
    # All of the paths used are relative, if we run the tests
    # from another directory we need to chdir()
    os.chdir(execution_path('.'))

wkts = [
    [mapnik.GeometryType.Point,
     "POINT(30 10)",
     "01010000000000000000003e400000000000002440"],
    [mapnik.GeometryType.Point,
     "POINT(30.0 10.0)",
     "01010000000000000000003e400000000000002440"],
    [mapnik.GeometryType.Point,
     "POINT(30.1 10.1)",
     "01010000009a99999999193e403333333333332440"],
    [mapnik.GeometryType.LineString,
     "LINESTRING(30 10,10 30,40 40)",
     "0102000000030000000000000000003e40000000000000244000000000000024400000000000003e4000000000000044400000000000004440"],
    [mapnik.GeometryType.Polygon,
     "POLYGON((30 10,10 20,20 40,40 40,30 10))",
     "010300000001000000050000000000000000003e4000000000000024400000000000002440000000000000344000000000000034400000000000004440000000000000444000000000000044400000000000003e400000000000002440"],
    [mapnik.GeometryType.Polygon,
     "POLYGON((35 10,10 20,15 40,45 45,35 10),(20 30,35 35,30 20,20 30))",
     "0103000000020000000500000000000000008041400000000000002440000000000000244000000000000034400000000000002e40000000000000444000000000008046400000000000804640000000000080414000000000000024400400000000000000000034400000000000003e40000000000080414000000000008041400000000000003e40000000000000344000000000000034400000000000003e40"],
    [mapnik.GeometryType.MultiPoint,
     "MULTIPOINT((10 40),(40 30),(20 20),(30 10))",
     "010400000004000000010100000000000000000024400000000000004440010100000000000000000044400000000000003e4001010000000000000000003440000000000000344001010000000000000000003e400000000000002440"],
    [mapnik.GeometryType.MultiLineString,
     "MULTILINESTRING((10 10,20 20,10 40),(40 40,30 30,40 20,30 10))",
     "010500000002000000010200000003000000000000000000244000000000000024400000000000003440000000000000344000000000000024400000000000004440010200000004000000000000000000444000000000000044400000000000003e400000000000003e40000000000000444000000000000034400000000000003e400000000000002440"],
    [mapnik.GeometryType.MultiPolygon,
     "MULTIPOLYGON(((30 20,10 40,45 40,30 20)),((15 5,40 10,10 20,5 10,15 5)))",
     "010600000002000000010300000001000000040000000000000000003e40000000000000344000000000000024400000000000004440000000000080464000000000000044400000000000003e400000000000003440010300000001000000050000000000000000002e4000000000000014400000000000004440000000000000244000000000000024400000000000003440000000000000144000000000000024400000000000002e400000000000001440"],
    [mapnik.GeometryType.MultiPolygon,
     "MULTIPOLYGON(((40 40,20 45,45 30,40 40)),((20 35,45 20,30 5,10 10,10 30,20 35),(30 20,20 25,20 15,30 20)))",
     "01060000000200000001030000000100000004000000000000000000444000000000000044400000000000003440000000000080464000000000008046400000000000003e40000000000000444000000000000044400103000000020000000600000000000000000034400000000000804140000000000080464000000000000034400000000000003e4000000000000014400000000000002440000000000000244000000000000024400000000000003e4000000000000034400000000000804140040000000000000000003e4000000000000034400000000000003440000000000000394000000000000034400000000000002e400000000000003e400000000000003440"],
    [mapnik.GeometryType.GeometryCollection,
     "GEOMETRYCOLLECTION(POLYGON((1 1,2 1,2 2,1 2,1 1)),POINT(2 3),LINESTRING(2 3,3 4))",
     "01070000000300000001030000000100000005000000000000000000f03f000000000000f03f0000000000000040000000000000f03f00000000000000400000000000000040000000000000f03f0000000000000040000000000000f03f000000000000f03f0101000000000000000000004000000000000008400102000000020000000000000000000040000000000000084000000000000008400000000000001040"],
    [mapnik.GeometryType.Polygon, "POLYGON((-178.32319 71.518365,-178.321586 71.518439,-178.259635 71.510688,-178.304862 71.513129,-178.32319 71.518365),(-178.32319 71.518365,-178.341544 71.517524,-178.32244 71.505439,-178.215323 71.478034,-178.193473 71.47663,-178.147757 71.485175,-178.124442 71.481879,-178.005729 71.448615,-178.017203 71.441413,-178.054191 71.428778,-178.047049 71.425727,-178.033439 71.417792,-178.026236 71.415107,-178.030082 71.413459,-178.039908 71.40766,-177.970878 71.39643,-177.779837 71.333197,-177.718375 71.305243,-177.706412 71.3039,-177.68212 71.304877,-177.670279 71.301825,-177.655387 71.293158,-177.587577 71.285956,-177.548575 71.294867,-177.531119 71.296332,-177.51409 71.293402,-177.498649 71.284735,-177.506217 71.268622,-177.486991 71.258734,-177.459708 71.249884,-177.443412 71.237006,-177.445914 71.222663,-177.457755 71.209357,-177.507804 71.173774,-177.581168 71.147589,-177.637626 71.117011,-177.684134 71.110968,-177.751883 71.092963,-177.819266 71.084662,-177.877677 71.052558,-177.930472 71.041449,-178.206595 71.038398,-178.310111 71.013617,-178.875907 70.981024,-178.980277 70.95069,-179.342093 70.908026,-179.336234 70.911078,-179.322257 70.921698,-179.364493 70.930243,-179.457511 70.915534,-179.501212 70.919684,-179.666007 70.965461,-179.853385 70.979438,-179.888785 70.993598,-179.907523 70.996772,-179.999989 70.992011,-179.999989 71.024848,-179.999989 71.058661,-179.999989 71.126166,-179.999989 71.187018,-179.999989 71.224189,-179.999989 71.27497,-179.999989 71.312079,-179.999989 71.356024,-179.999989 71.410041,-179.999989 71.487799,-179.999989 71.536689,-179.862845 71.538642,-179.912223 71.555854,-179.900748 71.558478,-179.798819 71.569098,-179.757438 71.583197,-179.735953 71.586432,-179.715445 71.583258,-179.697501 71.577338,-179.678702 71.573676,-179.610831 71.585211,-179.372062 71.569098,-179.326774 71.555487,-179.306815 71.557563,-179.287162 71.562934,-179.24285 71.569098,-179.204642 71.583197,-179.074576 71.600043,-178.395438 71.539008,-178.32319 71.518365))",
     "010300000002000000050000009e0c8e92574a66c0079964e42ce151403f1bb96e4a4a66c0247ec51a2ee15140b43c0fee4e4866c06b9db81cafe0514062f9f36dc14966c04568041bd7e051409e0c8e92574a66c0079964e42ce15140560000009e0c8e92574a66c0079964e42ce15140a4c4aeeded4a66c049b9fb1c1fe1514083ddb06d514a66c0dec7d11c59e0514074620fede34666c01118eb1b98de514017f549ee304666c0b8921d1b81de51402a58e36cba4466c08e75711b0ddf5140f607ca6dfb4366c04568041bd7de5140717495ee2e4066c0af5fb01bb6dc5140944c4eed8c4066c0dfc14f1c40dc51409a97c3eebb4166c0a2cf471971db5140e789e76c814166c08c81751c3fdb5140266daaee114166c0f321a81abdda51404b3fe1ecd64066c0ce55f31c91da514058c6866ef64066c029cfbc1c76da5140295b24ed464166c0cbf8f71917da5140fd4cbd6e113f66c0cf49ef1b5fd95140be66b96cf43866c00588821953d55140736891edfc3666c04f95ef1989d3514010b056ed9a3666c04850fc1873d35140527e52edd33566c05e13d21a83d351404206f2ec723566c05f07ce1951d35140276728eef83466c0e4a3c519c3d2514038dc476ecd3266c01406651a4dd25140e78c28ed8d3166c0db6ad619dfd2514018e945edfe3066c088307e1af7d251406c26df6c733066c02fa52e19c7d25140dc65bfeef42f66c0b341261939d25140bc75feed323066c01c3f541a31d15140484e266e952f66c040170d198fd05140b5368dedb52e66c0cbf27519fecf5140edd45c6e302e66c02bbd361b2bcf514083da6fed442e66c0dfc14f1c40ce51409352d0eda52e66c0130ce71a66cd5140e5982cee3f3066c049b9fb1c1fcb51406a2fa2ed983266c0f50f221972c951408a22a46e673466c06d91b41b7dc751409d82fc6ce43566c0c3b986191ac7514063edef6c0f3866c03c2f151bf3c551403197546d373a66c0b14d2a1a6bc55140726e13ee153c66c02ac93a1c5dc35140003b376dc63d66c0ecdcb419a7c251408c101e6d9c4666c0d68ee21c75c251402864e76dec4966c0db6ad619dfc05140cdea1d6e075c66c0d525e318c9be5140ac00df6d5e5f66c097a8de1ad8bc514019e6046df26a66c0bc7a15191dba5140b439ce6dc26a66c0ba86191a4fba5140077de9ed4f6a66c079b29b19fdba5140df5339eda96b66c04f95ef1989bb5140eed11beea36e66c01118eb1b98ba51409c8bbfed097066c0e3a9471adcba5140077de9ed4f7566c01024ef1ccabd5140b43c0fee4e7b66c06b9db81cafbe51409d853ded707c66c0bfd7101c97bf5140c5abac6d0a7d66c06364c91ccbbf5140836beee8ff7f66c06d91b41b7dbf5140836beee8ff7f66c0bfd7101c97c15140836beee8ff7f66c03e23111ac1c35140836beee8ff7f66c07ff78e1a13c85140836beee8ff7f66c0da70581af8cb5140836beee8ff7f66c0dec7d11c59ce5140836beee8ff7f66c06458c51b99d15140836beee8ff7f66c02db1321af9d35140836beee8ff7f66c0d525e318c9d65140836beee8ff7f66c03a419b1c3eda5140836beee8ff7f66c060014c1938df5140836beee8ff7f66c0dec7d11c59e251408c101e6d9c7b66c021904b1c79e2514017f549ee307d66c073d6a71c93e35140ff3d78edd27c66c04562821abee351408d0dddec8f7966c0048e041a6ce45140edd79dee3c7866c00588821953e55140944c4eed8c7766c0fc54151a88e55140c6a2e9ece47666c058c85c1954e55140acfd9ded517666c03c2f151bf3e451405ab741edb77566c002a08a1bb7e45140410c74ed8b7366c09b90d61874e55140b34291eee76b66c0048e041a6ce45140dc65bfee746a66c09a9658198de3514083ddb06dd16966c06b9db81cafe35140edd45c6e306966c09df3531c07e45140adfa5c6dc56766c0048e041a6ce451406b2c616d8c6666c00588821953e55140212235ed626266c0664cc11a67e6514010b3976da74c66c01212691b7fe251409e0c8e92574a66c0079964e42ce15140"],
    [mapnik.GeometryType.MultiPolygon, "MULTIPOLYGON(((-178.32319 71.518365,-178.321586 71.518439,-178.259635 71.510688,-178.304862 71.513129,-178.32319 71.518365)),((-178.32319 71.518365,-178.341544 71.517524,-178.32244 71.505439,-178.215323 71.478034,-178.193473 71.47663,-178.147757 71.485175,-178.124442 71.481879,-178.005729 71.448615,-178.017203 71.441413,-178.054191 71.428778,-178.047049 71.425727,-178.033439 71.417792,-178.026236 71.415107,-178.030082 71.413459,-178.039908 71.40766,-177.970878 71.39643,-177.779837 71.333197,-177.718375 71.305243,-177.706412 71.3039,-177.68212 71.304877,-177.670279 71.301825,-177.655387 71.293158,-177.587577 71.285956,-177.548575 71.294867,-177.531119 71.296332,-177.51409 71.293402,-177.498649 71.284735,-177.506217 71.268622,-177.486991 71.258734,-177.459708 71.249884,-177.443412 71.237006,-177.445914 71.222663,-177.457755 71.209357,-177.507804 71.173774,-177.581168 71.147589,-177.637626 71.117011,-177.684134 71.110968,-177.751883 71.092963,-177.819266 71.084662,-177.877677 71.052558,-177.930472 71.041449,-178.206595 71.038398,-178.310111 71.013617,-178.875907 70.981024,-178.980277 70.95069,-179.342093 70.908026,-179.336234 70.911078,-179.322257 70.921698,-179.364493 70.930243,-179.457511 70.915534,-179.501212 70.919684,-179.666007 70.965461,-179.853385 70.979438,-179.888785 70.993598,-179.907523 70.996772,-179.999989 70.992011,-179.999989 71.024848,-179.999989 71.058661,-179.999989 71.126166,-179.999989 71.187018,-179.999989 71.224189,-179.999989 71.27497,-179.999989 71.312079,-179.999989 71.356024,-179.999989 71.410041,-179.999989 71.487799,-179.999989 71.536689,-179.862845 71.538642,-179.912223 71.555854,-179.900748 71.558478,-179.798819 71.569098,-179.757438 71.583197,-179.735953 71.586432,-179.715445 71.583258,-179.697501 71.577338,-179.678702 71.573676,-179.610831 71.585211,-179.372062 71.569098,-179.326774 71.555487,-179.306815 71.557563,-179.287162 71.562934,-179.24285 71.569098,-179.204642 71.583197,-179.074576 71.600043,-178.395438 71.539008,-178.32319 71.518365)))",
     "010600000002000000010300000001000000050000009e0c8e92574a66c0079964e42ce151403f1bb96e4a4a66c0247ec51a2ee15140b43c0fee4e4866c06b9db81cafe0514062f9f36dc14966c04568041bd7e051409e0c8e92574a66c0079964e42ce15140010300000001000000560000009e0c8e92574a66c0079964e42ce15140a4c4aeeded4a66c049b9fb1c1fe1514083ddb06d514a66c0dec7d11c59e0514074620fede34666c01118eb1b98de514017f549ee304666c0b8921d1b81de51402a58e36cba4466c08e75711b0ddf5140f607ca6dfb4366c04568041bd7de5140717495ee2e4066c0af5fb01bb6dc5140944c4eed8c4066c0dfc14f1c40dc51409a97c3eebb4166c0a2cf471971db5140e789e76c814166c08c81751c3fdb5140266daaee114166c0f321a81abdda51404b3fe1ecd64066c0ce55f31c91da514058c6866ef64066c029cfbc1c76da5140295b24ed464166c0cbf8f71917da5140fd4cbd6e113f66c0cf49ef1b5fd95140be66b96cf43866c00588821953d55140736891edfc3666c04f95ef1989d3514010b056ed9a3666c04850fc1873d35140527e52edd33566c05e13d21a83d351404206f2ec723566c05f07ce1951d35140276728eef83466c0e4a3c519c3d2514038dc476ecd3266c01406651a4dd25140e78c28ed8d3166c0db6ad619dfd2514018e945edfe3066c088307e1af7d251406c26df6c733066c02fa52e19c7d25140dc65bfeef42f66c0b341261939d25140bc75feed323066c01c3f541a31d15140484e266e952f66c040170d198fd05140b5368dedb52e66c0cbf27519fecf5140edd45c6e302e66c02bbd361b2bcf514083da6fed442e66c0dfc14f1c40ce51409352d0eda52e66c0130ce71a66cd5140e5982cee3f3066c049b9fb1c1fcb51406a2fa2ed983266c0f50f221972c951408a22a46e673466c06d91b41b7dc751409d82fc6ce43566c0c3b986191ac7514063edef6c0f3866c03c2f151bf3c551403197546d373a66c0b14d2a1a6bc55140726e13ee153c66c02ac93a1c5dc35140003b376dc63d66c0ecdcb419a7c251408c101e6d9c4666c0d68ee21c75c251402864e76dec4966c0db6ad619dfc05140cdea1d6e075c66c0d525e318c9be5140ac00df6d5e5f66c097a8de1ad8bc514019e6046df26a66c0bc7a15191dba5140b439ce6dc26a66c0ba86191a4fba5140077de9ed4f6a66c079b29b19fdba5140df5339eda96b66c04f95ef1989bb5140eed11beea36e66c01118eb1b98ba51409c8bbfed097066c0e3a9471adcba5140077de9ed4f7566c01024ef1ccabd5140b43c0fee4e7b66c06b9db81cafbe51409d853ded707c66c0bfd7101c97bf5140c5abac6d0a7d66c06364c91ccbbf5140836beee8ff7f66c06d91b41b7dbf5140836beee8ff7f66c0bfd7101c97c15140836beee8ff7f66c03e23111ac1c35140836beee8ff7f66c07ff78e1a13c85140836beee8ff7f66c0da70581af8cb5140836beee8ff7f66c0dec7d11c59ce5140836beee8ff7f66c06458c51b99d15140836beee8ff7f66c02db1321af9d35140836beee8ff7f66c0d525e318c9d65140836beee8ff7f66c03a419b1c3eda5140836beee8ff7f66c060014c1938df5140836beee8ff7f66c0dec7d11c59e251408c101e6d9c7b66c021904b1c79e2514017f549ee307d66c073d6a71c93e35140ff3d78edd27c66c04562821abee351408d0dddec8f7966c0048e041a6ce45140edd79dee3c7866c00588821953e55140944c4eed8c7766c0fc54151a88e55140c6a2e9ece47666c058c85c1954e55140acfd9ded517666c03c2f151bf3e451405ab741edb77566c002a08a1bb7e45140410c74ed8b7366c09b90d61874e55140b34291eee76b66c0048e041a6ce45140dc65bfee746a66c09a9658198de3514083ddb06dd16966c06b9db81cafe35140edd45c6e306966c09df3531c07e45140adfa5c6dc56766c0048e041a6ce451406b2c616d8c6666c00588821953e55140212235ed626266c0664cc11a67e6514010b3976da74c66c01212691b7fe251409e0c8e92574a66c0079964e42ce15140"]
]


geojson = [
    [mapnik.GeometryType.Point, '{"type":"Point","coordinates":[30,10]}'],
    [mapnik.GeometryType.Point, '{"type":"Point","coordinates":[30.0,10.0]}'],
    [mapnik.GeometryType.Point, '{"type":"Point","coordinates":[30.1,10.1]}'],
    [mapnik.GeometryType.LineString,
     '{"type":"LineString","coordinates":[[30.0,10.0],[10.0,30.0],[40.0,40.0]]}'],
    [mapnik.GeometryType.Polygon,
     '{"type":"Polygon","coordinates":[[[30.0,10.0],[10.0,20.0],[20.0,40.0],[40.0,40.0],[30.0,10.0]]]}'],
    [mapnik.GeometryType.Polygon,
     '{"type":"Polygon","coordinates":[[[35.0,10.0],[10.0,20.0],[15.0,40.0],[45.0,45.0],[35.0,10.0]],[[20.0,30.0],[35.0,35.0],[30.0,20.0],[20.0,30.0]]]}'],
    [mapnik.GeometryType.MultiPoint,
     '{"type":"MultiPoint","coordinates":[[10.0,40.0],[40.0,30.0],[20.0,20.0],[30.0,10.0]]}'],
    [mapnik.GeometryType.MultiLineString,
     '{"type":"MultiLineString","coordinates":[[[10.0,10.0],[20.0,20.0],[10.0,40.0]],[[40.0,40.0],[30.0,30.0],[40.0,20.0],[30.0,10.0]]]}'],
    [mapnik.GeometryType.MultiPolygon,
     '{"type":"MultiPolygon","coordinates":[[[[30.0,20.0],[10.0,40.0],[45.0,40.0],[30.0,20.0]]],[[[15.0,5.0],[40.0,10.0],[10.0,20.0],[5.0,10.0],[15.0,5.0]]]]}'],
    [mapnik.GeometryType.MultiPolygon,
     '{"type":"MultiPolygon","coordinates":[[[[40.0,40.0],[20.0,45.0],[45.0,30.0],[40.0,40.0]]],[[[20.0,35.0],[45.0,20.0],[30.0,5.0],[10.0,10.0],[10.0,30.0],[20.0,35.0]],[[30.0,20.0],[20.0,25.0],[20.0,15.0],[30.0,20.0]]]]}'],
    [mapnik.GeometryType.GeometryCollection,
     '{"type":"GeometryCollection","geometries":[{"type":"Polygon","coordinates":[[[1.0,1.0],[2.0,1.0],[2.0,2.0],[1.0,2.0],[1.0,1.0]]]},{"type":"Point","coordinates":[2.0,3.0]},{"type":"LineString","coordinates":[[2.0,3.0],[3.0,4.0]]}]}'],
    [mapnik.GeometryType.Polygon, '{"type":"Polygon","coordinates":[[[-178.32319,71.518365],[-178.321586,71.518439],[-178.259635,71.510688],[-178.304862,71.513129],[-178.32319,71.518365]],[[-178.32319,71.518365],[-178.341544,71.517524],[-178.32244,71.505439],[-178.215323,71.478034],[-178.193473,71.47663],[-178.147757,71.485175],[-178.124442,71.481879],[-178.005729,71.448615],[-178.017203,71.441413],[-178.054191,71.428778],[-178.047049,71.425727],[-178.033439,71.417792],[-178.026236,71.415107],[-178.030082,71.413459],[-178.039908,71.40766],[-177.970878,71.39643],[-177.779837,71.333197],[-177.718375,71.305243],[-177.706412,71.3039],[-177.68212,71.304877],[-177.670279,71.301825],[-177.655387,71.293158],[-177.587577,71.285956],[-177.548575,71.294867],[-177.531119,71.296332],[-177.51409,71.293402],[-177.498649,71.284735],[-177.506217,71.268622],[-177.486991,71.258734],[-177.459708,71.249884],[-177.443412,71.237006],[-177.445914,71.222663],[-177.457755,71.209357],[-177.507804,71.173774],[-177.581168,71.147589],[-177.637626,71.117011],[-177.684134,71.110968],[-177.751883,71.092963],[-177.819266,71.084662],[-177.877677,71.052558],[-177.930472,71.041449],[-178.206595,71.038398],[-178.310111,71.013617],[-178.875907,70.981024],[-178.980277,70.95069],[-179.342093,70.908026],[-179.336234,70.911078],[-179.322257,70.921698],[-179.364493,70.930243],[-179.457511,70.915534],[-179.501212,70.919684],[-179.666007,70.965461],[-179.853385,70.979438],[-179.888785,70.993598],[-179.907523,70.996772],[-179.999989,70.992011],[-179.999989,71.024848],[-179.999989,71.058661],[-179.999989,71.126166],[-179.999989,71.187018],[-179.999989,71.224189],[-179.999989,71.27497],[-179.999989,71.312079],[-179.999989,71.356024],[-179.999989,71.410041],[-179.999989,71.487799],[-179.999989,71.536689],[-179.862845,71.538642],[-179.912223,71.555854],[-179.900748,71.558478],[-179.798819,71.569098],[-179.757438,71.583197],[-179.735953,71.586432],[-179.715445,71.583258],[-179.697501,71.577338],[-179.678702,71.573676],[-179.610831,71.585211],[-179.372062,71.569098],[-179.326774,71.555487],[-179.306815,71.557563],[-179.287162,71.562934],[-179.24285,71.569098],[-179.204642,71.583197],[-179.074576,71.600043],[-178.395438,71.539008],[-178.32319,71.518365]]]}'],
    [mapnik.GeometryType.MultiPolygon, '{"type":"MultiPolygon","coordinates":[[[[-178.32319,71.518365],[-178.321586,71.518439],[-178.259635,71.510688],[-178.304862,71.513129],[-178.32319,71.518365]]],[[[-178.32319,71.518365],[-178.341544,71.517524],[-178.32244,71.505439],[-178.215323,71.478034],[-178.193473,71.47663],[-178.147757,71.485175],[-178.124442,71.481879],[-178.005729,71.448615],[-178.017203,71.441413],[-178.054191,71.428778],[-178.047049,71.425727],[-178.033439,71.417792],[-178.026236,71.415107],[-178.030082,71.413459],[-178.039908,71.40766],[-177.970878,71.39643],[-177.779837,71.333197],[-177.718375,71.305243],[-177.706412,71.3039],[-177.68212,71.304877],[-177.670279,71.301825],[-177.655387,71.293158],[-177.587577,71.285956],[-177.548575,71.294867],[-177.531119,71.296332],[-177.51409,71.293402],[-177.498649,71.284735],[-177.506217,71.268622],[-177.486991,71.258734],[-177.459708,71.249884],[-177.443412,71.237006],[-177.445914,71.222663],[-177.457755,71.209357],[-177.507804,71.173774],[-177.581168,71.147589],[-177.637626,71.117011],[-177.684134,71.110968],[-177.751883,71.092963],[-177.819266,71.084662],[-177.877677,71.052558],[-177.930472,71.041449],[-178.206595,71.038398],[-178.310111,71.013617],[-178.875907,70.981024],[-178.980277,70.95069],[-179.342093,70.908026],[-179.336234,70.911078],[-179.322257,70.921698],[-179.364493,70.930243],[-179.457511,70.915534],[-179.501212,70.919684],[-179.666007,70.965461],[-179.853385,70.979438],[-179.888785,70.993598],[-179.907523,70.996772],[-179.999989,70.992011],[-179.999989,71.024848],[-179.999989,71.058661],[-179.999989,71.126166],[-179.999989,71.187018],[-179.999989,71.224189],[-179.999989,71.27497],[-179.999989,71.312079],[-179.999989,71.356024],[-179.999989,71.410041],[-179.999989,71.487799],[-179.999989,71.536689],[-179.862845,71.538642],[-179.912223,71.555854],[-179.900748,71.558478],[-179.798819,71.569098],[-179.757438,71.583197],[-179.735953,71.586432],[-179.715445,71.583258],[-179.697501,71.577338],[-179.678702,71.573676],[-179.610831,71.585211],[-179.372062,71.569098],[-179.326774,71.555487],[-179.306815,71.557563],[-179.287162,71.562934],[-179.24285,71.569098],[-179.204642,71.583197],[-179.074576,71.600043],[-178.395438,71.539008],[-178.32319,71.518365]]]]}']
]

geojson_reversed = [
    '{"coordinates":[30,10],"type":"Point"}',
    '{"coordinates":[30.0,10.0],"type":"Point"}',
    '{"coordinates":[30.1,10.1],"type":"Point"}',
    '{"coordinates":[[30.0,10.0],[10.0,30.0],[40.0,40.0]],"type":"LineString"}',
    '{"coordinates":[[[30.0,10.0],[10.0,20.0],[20.0,40.0],[40.0,40.0],[30.0,10.0]]],"type":"Polygon"}',
    '{"coordinates":[[[35.0,10.0],[10.0,20.0],[15.0,40.0],[45.0,45.0],[35.0,10.0]],[[20.0,30.0],[35.0,35.0],[30.0,20.0],[20.0,30.0]]],"type":"Polygon"}',
    '{"coordinates":[[10.0,40.0],[40.0,30.0],[20.0,20.0],[30.0,10.0]],"type":"MultiPoint"}',
    '{"coordinates":[[[10.0,10.0],[20.0,20.0],[10.0,40.0]],[[40.0,40.0],[30.0,30.0],[40.0,20.0],[30.0,10.0]]],"type":"MultiLineString"}',
    '{"coordinates":[[[[30.0,20.0],[10.0,40.0],[45.0,40.0],[30.0,20.0]]],[[[15.0,5.0],[40.0,10.0],[10.0,20.0],[5.0,10.0],[15.0,5.0]]]],"type":"MultiPolygon"}',
    '{"coordinates":[[[[40.0,40.0],[20.0,45.0],[45.0,30.0],[40.0,40.0]]],[[[20.0,35.0],[45.0,20.0],[30.0,5.0],[10.0,10.0],[10.0,30.0],[20.0,35.0]],[[30.0,20.0],[20.0,25.0],[20.0,15.0],[30.0,20.0]]]],"type":"MultiPolygon"}',
    '{"geometries":[{"coordinates":[[[1.0,1.0],[2.0,1.0],[2.0,2.0],[1.0,2.0],[1.0,1.0]]],"type":"Polygon"},{"coordinates":[2.0,3.0],"type":"Point"},{"coordinates":[[2.0,3.0],[3.0,4.0]],"type":"LineString"}],"type":"GeometryCollection"}',
    '{"coordinates":[[[-178.32319,71.518365],[-178.321586,71.518439],[-178.259635,71.510688],[-178.304862,71.513129],[-178.32319,71.518365]],[[-178.32319,71.518365],[-178.341544,71.517524],[-178.32244,71.505439],[-178.215323,71.478034],[-178.193473,71.47663],[-178.147757,71.485175],[-178.124442,71.481879],[-178.005729,71.448615],[-178.017203,71.441413],[-178.054191,71.428778],[-178.047049,71.425727],[-178.033439,71.417792],[-178.026236,71.415107],[-178.030082,71.413459],[-178.039908,71.40766],[-177.970878,71.39643],[-177.779837,71.333197],[-177.718375,71.305243],[-177.706412,71.3039],[-177.68212,71.304877],[-177.670279,71.301825],[-177.655387,71.293158],[-177.587577,71.285956],[-177.548575,71.294867],[-177.531119,71.296332],[-177.51409,71.293402],[-177.498649,71.284735],[-177.506217,71.268622],[-177.486991,71.258734],[-177.459708,71.249884],[-177.443412,71.237006],[-177.445914,71.222663],[-177.457755,71.209357],[-177.507804,71.173774],[-177.581168,71.147589],[-177.637626,71.117011],[-177.684134,71.110968],[-177.751883,71.092963],[-177.819266,71.084662],[-177.877677,71.052558],[-177.930472,71.041449],[-178.206595,71.038398],[-178.310111,71.013617],[-178.875907,70.981024],[-178.980277,70.95069],[-179.342093,70.908026],[-179.336234,70.911078],[-179.322257,70.921698],[-179.364493,70.930243],[-179.457511,70.915534],[-179.501212,70.919684],[-179.666007,70.965461],[-179.853385,70.979438],[-179.888785,70.993598],[-179.907523,70.996772],[-179.999989,70.992011],[-179.999989,71.024848],[-179.999989,71.058661],[-179.999989,71.126166],[-179.999989,71.187018],[-179.999989,71.224189],[-179.999989,71.27497],[-179.999989,71.312079],[-179.999989,71.356024],[-179.999989,71.410041],[-179.999989,71.487799],[-179.999989,71.536689],[-179.862845,71.538642],[-179.912223,71.555854],[-179.900748,71.558478],[-179.798819,71.569098],[-179.757438,71.583197],[-179.735953,71.586432],[-179.715445,71.583258],[-179.697501,71.577338],[-179.678702,71.573676],[-179.610831,71.585211],[-179.372062,71.569098],[-179.326774,71.555487],[-179.306815,71.557563],[-179.287162,71.562934],[-179.24285,71.569098],[-179.204642,71.583197],[-179.074576,71.600043],[-178.395438,71.539008],[-178.32319,71.518365]]],"type":"Polygon"}',
    '{"coordinates":[[[[-178.32319,71.518365],[-178.321586,71.518439],[-178.259635,71.510688],[-178.304862,71.513129],[-178.32319,71.518365]]],[[[-178.32319,71.518365],[-178.341544,71.517524],[-178.32244,71.505439],[-178.215323,71.478034],[-178.193473,71.47663],[-178.147757,71.485175],[-178.124442,71.481879],[-178.005729,71.448615],[-178.017203,71.441413],[-178.054191,71.428778],[-178.047049,71.425727],[-178.033439,71.417792],[-178.026236,71.415107],[-178.030082,71.413459],[-178.039908,71.40766],[-177.970878,71.39643],[-177.779837,71.333197],[-177.718375,71.305243],[-177.706412,71.3039],[-177.68212,71.304877],[-177.670279,71.301825],[-177.655387,71.293158],[-177.587577,71.285956],[-177.548575,71.294867],[-177.531119,71.296332],[-177.51409,71.293402],[-177.498649,71.284735],[-177.506217,71.268622],[-177.486991,71.258734],[-177.459708,71.249884],[-177.443412,71.237006],[-177.445914,71.222663],[-177.457755,71.209357],[-177.507804,71.173774],[-177.581168,71.147589],[-177.637626,71.117011],[-177.684134,71.110968],[-177.751883,71.092963],[-177.819266,71.084662],[-177.877677,71.052558],[-177.930472,71.041449],[-178.206595,71.038398],[-178.310111,71.013617],[-178.875907,70.981024],[-178.980277,70.95069],[-179.342093,70.908026],[-179.336234,70.911078],[-179.322257,70.921698],[-179.364493,70.930243],[-179.457511,70.915534],[-179.501212,70.919684],[-179.666007,70.965461],[-179.853385,70.979438],[-179.888785,70.993598],[-179.907523,70.996772],[-179.999989,70.992011],[-179.999989,71.024848],[-179.999989,71.058661],[-179.999989,71.126166],[-179.999989,71.187018],[-179.999989,71.224189],[-179.999989,71.27497],[-179.999989,71.312079],[-179.999989,71.356024],[-179.999989,71.410041],[-179.999989,71.487799],[-179.999989,71.536689],[-179.862845,71.538642],[-179.912223,71.555854],[-179.900748,71.558478],[-179.798819,71.569098],[-179.757438,71.583197],[-179.735953,71.586432],[-179.715445,71.583258],[-179.697501,71.577338],[-179.678702,71.573676],[-179.610831,71.585211],[-179.372062,71.569098],[-179.326774,71.555487],[-179.306815,71.557563],[-179.287162,71.562934],[-179.24285,71.569098],[-179.204642,71.583197],[-179.074576,71.600043],[-178.395438,71.539008],[-178.32319,71.518365]]]],"type":"MultiPolygon"}'
]

valid_empty_geometries = [
    'null', # Point can't be empty
    '{ "type": "LineString" , "coordinates": []}',
    '{ "type": "Polygon" , "coordinates": [[]]}',
    '{ "type": "MultiPoint" , "coordinates": []}',
    '{ "type": "MultiLineString" , "coordinates": [[]]}',
    '{ "type": "MultiPolygon" , "coordinates": [[[]]]}',
]

invalid_empty_geometries = [
    '{ "type": "Point" , "coordinates": []}', # Point can't be empty
    '{ "type": "LineString" , "coordinates": [[]]}',
    '{ "type": "Polygon" , "coordinates": [[[]]]}',
    '{ "type": "MultiPoint" , "coordinates": [[]]}',
    '{ "type": "MultiLineString" , "coordinates": [[[]]]}',
    '{ "type": "MultiPolygon" , "coordinates": [[[[]]]]}',
]


# valid, but empty wkb's
# (http://trac.osgeo.org/postgis/wiki/DevWikiEmptyGeometry)
empty_wkbs = [
    # TODO - this is messed up: round trips as MULTIPOINT EMPTY
    # template_postgis=# select ST_AsText(ST_GeomFromEWKB(decode(encode(ST_GeomFromText('POINT EMPTY'),'hex'),'hex')));
    #    st_astext
    #------------------
    # MULTIPOINT EMPTY
    #(1 row)
    #[ mapnik.GeometryType.Point,              "Point EMPTY", '010400000000000000'],
    [mapnik.GeometryType.MultiPoint, "MULTIPOINT EMPTY", '010400000000000000'],
    [mapnik.GeometryType.LineString, "LINESTRING EMPTY", '010200000000000000'],
    [mapnik.GeometryType.LineString, "LINESTRING EMPTY", '010200000000000000'],
    [mapnik.GeometryType.MultiLineString,
     "MULTILINESTRING EMPTY",
     '010500000000000000'],
    [mapnik.GeometryType.Polygon, "Polygon EMPTY", '010300000000000000'],
    [mapnik.GeometryType.GeometryCollection,
        "GEOMETRYCOLLECTION EMPTY", '010700000000000000'],
    [mapnik.GeometryType.GeometryCollection,
     "GEOMETRYCOLLECTION(LINESTRING EMPTY,LINESTRING EMPTY)",
     '010700000000000000'],
    [mapnik.GeometryType.GeometryCollection,
     "GEOMETRYCOLLECTION(POINT EMPTY,POINT EMPTY)",
     '010700000000000000'],
]

partially_empty_wkb = [
    # TODO - currently this is not considered empty
    # even though one part is
    [mapnik.GeometryType.GeometryCollection,
     "GEOMETRYCOLLECTION(MULTILINESTRING((10 10,20 20,10 40),(40 40,30 30,40 20,30 10)),LINESTRING EMPTY)",
     '010700000002000000010500000002000000010200000003000000000000000000244000000000000024400000000000003440000000000000344000000000000024400000000000004440010200000004000000000000000000444000000000000044400000000000003e400000000000003e40000000000000444000000000000034400000000000003e400000000000002440010200000000000000'],
    [mapnik.GeometryType.GeometryCollection,
     "GEOMETRYCOLLECTION(POINT EMPTY,POINT(0 0))",
     '010700000002000000010400000000000000010100000000000000000000000000000000000000'],
    [mapnik.GeometryType.GeometryCollection,
     "GEOMETRYCOLLECTION(POINT EMPTY,MULTIPOINT(0 0))",
     '010700000002000000010400000000000000010400000001000000010100000000000000000000000000000000000000'],
]

# unsupported types
unsupported_wkb = [
    ["MULTIPOLYGON EMPTY", '010600000000000000'],
    ["TRIANGLE EMPTY", '011100000000000000'],
    ["CircularString EMPTY", '010800000000000000'],
    ["CurvePolygon EMPTY", '010A00000000000000'],
    ["CompoundCurve EMPTY", '010900000000000000'],
    ["MultiCurve EMPTY", '010B00000000000000'],
    ["MultiSurface EMPTY", '010C00000000000000'],
    ["PolyhedralSurface EMPTY", '010F00000000000000'],
    ["TIN EMPTY", '011000000000000000'],
    # TODO - a few bogus inputs
    # enable if we start range checking to avoid crashing on invalid input?
    # https://github.com/mapnik/mapnik/issues/2236
    #[ "", '' ],
    #[ "00", '01' ],
    #[ "0000", '0104' ],
]


def test_path_geo_interface():
    geom = mapnik.Geometry.from_wkt('POINT(0 0)')
    eq_(geom.__geo_interface__, {u'type': u'Point', u'coordinates': [0, 0]})


def test_valid_wkb_parsing():
    count = 0
    for wkb in empty_wkbs:
        geom = mapnik.Geometry.from_wkb(unhexlify(wkb[2]))
        eq_(geom.is_empty(), True)
        eq_(geom.type(), wkb[0])

    for wkb in wkts:
        geom = mapnik.Geometry.from_wkb(unhexlify(wkb[2]))
        eq_(geom.is_empty(), False)
        eq_(geom.type(), wkb[0])


def test_wkb_parsing_error():
    count = 0
    for wkb in unsupported_wkb:
        try:
            geom = mapnik.Geometry.from_wkb(unhexlify(wkb))
            # should not get here
            eq_(True, False)
        except:
            pass
    assert True

# for partially empty wkbs don't currently look empty right now
# since the enclosing container has objects


def test_empty_wkb_parsing():
    count = 0
    for wkb in partially_empty_wkb:
        geom = mapnik.Geometry.from_wkb(unhexlify(wkb[2]))
        eq_(geom.type(), wkb[0])
        eq_(geom.is_empty(), False)


def test_geojson_parsing():
    geometries = []
    count = 0
    for j in geojson:
        count += 1
        geometries.append(mapnik.Geometry.from_geojson(j[1]))
    eq_(count, len(geometries))


def test_geojson_parsing_reversed():
    for idx, j in enumerate(geojson_reversed):
        g1 = mapnik.Geometry.from_geojson(j)
        g2 = mapnik.Geometry.from_geojson(geojson[idx][1])
        eq_(g1.to_geojson(), g2.to_geojson())

# http://geojson.org/geojson-spec.html#positions


def test_geojson_point_positions():
    input_json = '{"type":"Point","coordinates":[30,10]}'
    geom = mapnik.Geometry.from_geojson(input_json)
    eq_(geom.to_geojson(), input_json)
    # should ignore all but the first two
    geom = mapnik.Geometry.from_geojson(
        '{"type":"Point","coordinates":[30,10,50,50,50,50]}')
    eq_(geom.to_geojson(), input_json)


def test_geojson_point_positions2():
    input_json = '{"type":"LineString","coordinates":[[30,10],[10,30],[40,40]]}'
    geom = mapnik.Geometry.from_geojson(input_json)
    eq_(geom.to_geojson(), input_json)

    # should ignore all but the first two
    geom = mapnik.Geometry.from_geojson(
        '{"type":"LineString","coordinates":[[30.0,10.0,0,0,0],[10.0,30.0,0,0,0],[40.0,40.0,0,0,0]]}')
    eq_(geom.to_geojson(), input_json)


def compare_wkb_from_wkt(wkt, type):
    geom = mapnik.Geometry.from_wkt(wkt)
    eq_(geom.type(), type)


def compare_wkt_to_geojson(idx, wkt, num=None):
    geom = mapnik.Geometry.from_wkt(wkt)
    # ensure both have same result
    gj = geom.to_geojson()
    eq_(len(gj) > 1, True)
    a = json.loads(gj)
    e = json.loads(geojson[idx][1])
    eq_(a, e)


def test_wkt_simple():
    for wkt in wkts:
        try:
            geom = mapnik.Geometry.from_wkt(wkt[1])
            eq_(geom.type(), wkt[0])
        except RuntimeError as e:
            raise RuntimeError('%s %s' % (e, wkt))


def test_wkb_simple():
    for wkt in wkts:
        try:
            compare_wkb_from_wkt(wkt[1], wkt[0])
        except RuntimeError as e:
            raise RuntimeError('%s %s' % (e, wkt))


def test_wkt_to_geojson():
    idx = -1
    for wkt in wkts:
        try:
            idx += 1
            compare_wkt_to_geojson(idx, wkt[1], wkt[0])
        except RuntimeError as e:
            raise RuntimeError('%s %s' % (e, wkt))


def test_wkt_rounding():
    # currently fails because we use output precision of 6 - should we make configurable? https://github.com/mapnik/mapnik/issues/1009
    # if precision is set to 15 still fails due to very subtle rounding issues
    wkt = "POLYGON((7.904185 54.180426,7.89918 54.178168,7.897715 54.182318,7.893565 54.183111,7.890391 54.187567,7.885874 54.19068,7.879893 54.193915,7.894541 54.194647,7.900645 54.19068,7.904185 54.180426))"
    geom = mapnik.Geometry.from_wkt(wkt)
    eq_(geom.type(), mapnik.GeometryType.Polygon)


def test_wkt_collection_flattening():
    wkt = 'GEOMETRYCOLLECTION(POLYGON((1 1,2 1,2 2,1 2,1 1)),POLYGON((40 40,20 45,45 30,40 40)),POLYGON((20 35,45 20,30 5,10 10,10 30,20 35),(30 20,20 25,20 15,30 20)),LINESTRING(2 3,3 4))'
    # currently fails as the MULTIPOLYGON inside will be returned as multiple polygons - not a huge deal - should we worry?
    #wkt = "GEOMETRYCOLLECTION(POLYGON((1 1,2 1,2 2,1 2,1 1)),MULTIPOLYGON(((40 40,20 45,45 30,40 40)),((20 35,45 20,30 5,10 10,10 30,20 35),(30 20,20 25,20 15,30 20))),LINESTRING(2 3,3 4))"
    geom = mapnik.Geometry.from_wkt(wkt)
    eq_(geom.type(), mapnik.GeometryType.GeometryCollection)


def test_creating_feature_from_geojson():
    json_feat = {
        "type": "Feature",
        "geometry": {"type": "Point", "coordinates": [-122, 48]},
        "properties": {"name": "value"}
    }
    ctx = mapnik.Context()
    feat = mapnik.Feature.from_geojson(json.dumps(json_feat), ctx)
    eq_(feat.id(), 1)
    eq_(feat['name'], u'value')


def test_handling_valid_geojson_empty_geometries():
    for json in valid_empty_geometries:
        geom = mapnik.Geometry.from_geojson(json)
        out_json = geom.to_geojson()
        # check round trip
        eq_(json.replace(" ",""), out_json)


def test_handling_invalid_geojson_empty_geometries():
    for json in invalid_empty_geometries:
        assert_raises(RuntimeError, mapnik.Geometry.from_geojson, json)

if __name__ == "__main__":
    setup()
    exit(run_all(eval(x) for x in dir() if x.startswith("test_")))
