/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include <xine/sorted_array.h>

#include "_font.h"

#include "_xitk.h"
#include "_backend.h"

#define XITK_CACHE_SIZE 10

typedef struct {
  xitk_dnode_t node;
  xitk_font_cache_t *fc;
  xitk_font_t *font;
  int          refs;
  char         load_name[1];
} xitk_font_cache_item_t;

struct xitk_font_cache_s {
  xitk_dlist_t list; /** << self organizing list with used items going to top. */
  xine_sarray_t *index; /** << by load name. */
  xitk_t *xitk;
  int nused, nfree;
  pthread_mutex_t mutex;
};

xitk_font_cache_t *xitk_font_cache_init (void) {
  xitk_font_cache_t *fc = malloc (sizeof (*fc));
  xine_sarray_t *a = xine_sarray_new (32, (int (*)(void *, void *))strcasecmp);

  if (fc && a) {
#ifdef XINE_SARRAY_MODE_UNIQUE
    xine_sarray_set_mode (a, XINE_SARRAY_MODE_UNIQUE);
#endif
#if defined(XINE_SARRAY) && (XINE_SARRAY >= 3)
    xine_sarray_set_hash (a, (unsigned int (*)(void *))xitk_string_hash_32, 32);
#endif
    xitk_dlist_init (&fc->list);
    fc->index = a;
    fc->xitk = NULL;
    fc->nused = fc->nfree = 0;
    pthread_mutex_init (&fc->mutex, NULL);
    return fc;
  }

  xine_sarray_delete (a);
  free (fc);
  return NULL;
}

void xitk_font_cache_destroy (xitk_font_cache_t **p) {
  xitk_font_cache_t *fc;

  if (!p)
    return;
  fc = *p;
  if (!fc)
    return;
  *p = NULL;
  pthread_mutex_lock (&fc->mutex);
  if (fc->xitk && (fc->xitk->verbosity >= 2)) {
#if defined(XINE_SARRAY) && (XINE_SARRAY >= 4)
    unsigned int q = xine_sarray_hash_quality (fc->index);
    printf ("xitk.font_cache.hash_quality (%u.%u%%).\n", q / 10u, q % 10u);
#endif
    printf ("yitk.font.cache.delete (%d used, %d free).\n", fc->nused, fc->nfree);
  }
  xine_sarray_delete (fc->index);
  do {
    xitk_font_cache_item_t *item;
    xitk_container (item, fc->list.head.next, node);
    if (!item->node.next)
      break;
    xitk_dnode_remove (&item->node);
    item->font->_delete (&item->font);
    free (item);
  } while (0);
  pthread_mutex_unlock (&fc->mutex);
  pthread_mutex_destroy (&fc->mutex);
  free (fc);
}

xitk_font_t *xitk_font_load_font (xitk_t *xitk, const char *name) {
  xitk_font_cache_t *fc;
  xitk_font_cache_item_t *item;
  size_t l;
  int i;

  if (!xitk || !name)
    return NULL;

  fc = xitk->font_cache;
  pthread_mutex_lock (&fc->mutex);
  fc->xitk = xitk;

#if defined(XINE_SARRAY_MODE_UNIQUE) && defined(XINE_SARRAY) && (XINE_SARRAY >= 2)
  i = ~xine_sarray_add (fc->index, (void *)name);
#else
  i = xine_sarray_binary_search (fc->index, (void *)name);
#endif
  if (i >= 0) {
    char *load_name = xine_sarray_get (fc->index, i);
    xitk_container (item, load_name, load_name[0]);
    if (++item->refs == 1)
      fc->nused++, fc->nfree--;
    if (&item->node != fc->list.head.next) {
      xitk_dnode_remove (&item->node);
      xitk_dlist_add_head (&fc->list, &item->node);
    }
    pthread_mutex_unlock (&fc->mutex);
    return item->font;
  }

  l = xitk_find_byte (name, 0);
  item = malloc (xitk_offsetof (xitk_font_cache_item_t, load_name) + l + 1);
#if defined(XINE_SARRAY_MODE_UNIQUE) && defined(XINE_SARRAY) && (XINE_SARRAY >= 2)
  xine_sarray_move_location (fc->index, item->load_name, ~i);
#endif
  if (!item) {
    pthread_mutex_unlock (&fc->mutex);
    return NULL;
  }

  xitk_dnode_init (&item->node);
  item->fc = fc;
  memcpy (item->load_name, name, l + 1);
  item->refs = 1;
    
  /* font not found, load it */
  item->font = xitk->d->font_new (xitk->d, name);
  if (!item->font) {
    const char *fdname = xitk_get_cfg_string (xitk, XITK_DEFAULT_FONT);

    item->font = fdname ? xitk->d->font_new (xitk->d, fdname) : NULL;
    if (!item->font) {
      const char *fsname = xitk_get_cfg_string (xitk, XITK_SYSTEM_FONT);

      item->font = xitk->d->font_new (xitk->d, fsname);
      if (!item->font) {
        XITK_WARNING ("loading font \"%s\" failed, default and system fonts \"%s\" and \"%s\" failed too\n", name, fdname, fsname);
        pthread_mutex_unlock (&fc->mutex);

        /* Maybe broken XMB support */
        if (xitk_get_cfg_num (xitk, XITK_XMB_ENABLE)) {
          xitk_set_xmb_enability (xitk, 0);
          if ((item->font = xitk_font_load_font (xitk, name)))
            XITK_WARNING ("XMB support seems broken on your system, add \"font.xmb = 0\" in your ~/.xitkrc\n");
        }
      }
    }
  }
  if (!item->font) {
#if defined(XINE_SARRAY_MODE_UNIQUE) && defined(XINE_SARRAY) && (XINE_SARRAY >= 2)
    xine_sarray_move_location (fc->index, NULL, ~i);
#endif
    pthread_mutex_unlock (&fc->mutex);
    free (item);
    return NULL;
  }

  item->font->data = item;
  xitk_dlist_add_head (&fc->list, &item->node);
#if defined(XINE_SARRAY_MODE_UNIQUE) && defined(XINE_SARRAY) && (XINE_SARRAY >= 2)
#else
  xine_sarray_add (fc->index, item->load_name);
#endif
  fc->nused++;
  if (fc->xitk->verbosity >= 2)
    printf ("xiTK.font.load.new (%s) = %d used, %d free.\n", name, fc->nused, fc->nfree);
  pthread_mutex_unlock (&fc->mutex);
  return item->font;
}

void xitk_font_unload_font (xitk_font_t *font) {
  xitk_font_cache_t *fc;
  xitk_font_cache_item_t *item;

  if (!font)
    return;
  item = font->data;
  if (!item)
    return;
  if (item->font != font)
    return;
  fc = item->fc;
  if (!fc)
    return;
  pthread_mutex_lock (&fc->mutex);
  if (--item->refs == 0) {
    fc->nused--;
    if (++fc->nfree > XITK_CACHE_SIZE) {
      xitk_container (item, fc->list.tail.prev, node);
      while (item->node.prev) {
        if (item->refs <= 0) {
#ifdef XINE_SARRAY_MODE_UNIQUE
          xine_sarray_remove_ptr (fc->index, item->load_name);
#else
          xine_sarray_remove (fc->index, xine_sarray_binary_search (fc->index, item->load_name));
#endif
          xitk_dnode_remove (&item->node);
          item->font->_delete (&item->font);
          free (item);
          fc->nfree--;
          break;
        }
        xitk_container (item, item->node.prev, node);
      }
    }
  }
  pthread_mutex_unlock (&fc->mutex);
}


/*
 *
 */
int xitk_font_text_width (xitk_font_t *xtfs, const char *text, int nbytes) {
  xitk_font_textent_t ext;

  xitk_font_text_extent (xtfs, text, nbytes, &ext);
  return ext.width;
}

/*
 *
 */
int xitk_font_text_height (xitk_font_t *xtfs, const char *text, int nbytes) {
  xitk_font_textent_t ext;

  xitk_font_text_extent (xtfs, text, nbytes, &ext);
  return ext.ascent + ext.descent;
}

/*
 *
 */
size_t xitk_font_text_extent (xitk_font_t *xtfs, const char *text, int nbytes, xitk_font_textent_t *ext) {

  if (!xtfs || !text || !nbytes) {
    if (ext)
      ext->width = ext->ascent = ext->descent = ext->lbearing = ext->rbearing = 0;
    return 0;
  }
  if (nbytes == -1)
    nbytes = xitk_find_byte (text, 0);

#ifdef DEBUG
  if (nbytes > strlen (text) + 1) {
    XITK_WARNING ("extent: %zu > %zu\n", nbytes, strlen (text));
  }
#endif

  xtfs->text_extent (xtfs, text, nbytes, ext);
  return nbytes;
}
