# make_stringr_plugin.R
# This script programmatically generates an RKWard plugin for stringr string manipulation.
# To run: source("make_stringr_plugin.R")

local({
  # =========================================================================================
  # Package Definition and Metadata
  # =========================================================================================
  library(rkwarddev)
  rkwarddev.required("0.08-1")

  package_about <- rk.XML.about(
    name = "rk.stringr",
    author = person(
      given = "Alfonso",
      family = "Cano",
      email = "alfonso.cano@correo.buap.mx",
      role = c("aut", "cre")
    ),
    about = list(
      desc = "An RKWard plugin for string manipulation using the 'stringr' library.",
      version = "0.1.0",
      url = "https://github.com/AlfCano/", # Placeholder URL
      license = "GPL (>= 3)"
    )
  )

  dependencies.info <- rk.XML.dependencies(
    dependencies = list(
      rkward.min = "0.7.0",
      R.min = "3.5.0",
      package = list(name = "stringr", min = "1.4.0", repository = "CRAN")
    )
  )

  # =========================================================================================
  # Main Plugin: Detect Matches
  # =========================================================================================
  help_main <- rk.rkh.doc(
    title = rk.rkh.title(text = "Detect & Locate Matches"),
    summary = rk.rkh.summary(text = "Detect, locate, or count pattern matches in a character vector.")
  )

  detect_selector <- rk.XML.varselector(id.name="detect_selector")
  detect_slot <- rk.XML.varslot("String or vector to search", source="detect_selector", required=TRUE, id.name="detect_string")
  detect_pattern <- rk.XML.input("Pattern (regular expression)", id.name="detect_pattern")

  detect_dropdown <- rk.XML.dropdown(label = "Function", options = list(
    "Detect matches (TRUE/FALSE)" = list(val = "str_detect", chk = TRUE),
    "Count number of matches" = list(val = "str_count"),
    "Locate first match position" = list(val = "str_locate"),
    "Locate all match positions" = list(val = "str_locate_all"),
    "Check if string starts with pattern" = list(val = "str_starts"),
    "Check if string ends with pattern" = list(val = "str_ends")
  ), id.name = "detect_func")

  detect_save_obj <- rk.XML.saveobj("Save result as", initial="stringr.detected", chk=TRUE, id.name="detect_save")
  detect_preview <- rk.XML.preview(mode = "data")

  # CORRECTED: Two-column layout
  main_dialog <- rk.XML.dialog(label="Detect & Locate Matches", child=rk.XML.row(
      detect_selector,
      rk.XML.col(
          detect_slot,
          detect_pattern,
          rk.XML.frame(detect_dropdown, label="Detection Function"),
          detect_save_obj,
          detect_preview
      )
  ))

  js_calc_main <- "
    var string_vec = getValue('detect_string');
    var pattern = getValue('detect_pattern');
    var operation = getValue('detect_func');
    echo('stringr.detected <- ' + operation + '(string=' + string_vec + ', pattern=\"' + pattern + '\")\\n');
  "
  js_preview_main <- "
    var string_vec = getValue('detect_string');
    var pattern = getValue('detect_pattern');
    var operation = getValue('detect_func');
    echo('preview_data <- as.data.frame(' + operation + '(string=' + string_vec + ', pattern=\"' + pattern + '\"))\\n');
  "
  js_print_main <- "
    var save_name = getValue('detect_save.objectname');
    if(getValue('detect_save.active')){
      echo('rk.header(\"Detection result saved as: ' + save_name + '\", level=3)\\n');
    }
  "

  # =========================================================================================
  # Component 1: Mutate Strings
  # =========================================================================================
  mutate_selector <- rk.XML.varselector(id.name="mutate_selector")
  mutate_slot <- rk.XML.varslot("String or vector to mutate", source="mutate_selector", required=TRUE, id.name="mutate_string")

  mutate_dropdown <- rk.XML.dropdown(label="Function", options=list(
      "Convert to lower case"=list(val="str_to_lower", chk=TRUE),
      "Convert to upper case"=list(val="str_to_upper"),
      "Convert to title case"=list(val="str_to_title"),
      "Convert to sentence case"=list(val="str_to_sentence")
  ), id.name="mutate_func")

  mutate_save_obj <- rk.XML.saveobj("Save result as", initial="stringr.mutated", chk=TRUE, id.name="mutate_save")
  mutate_preview <- rk.XML.preview(mode = "data")

  # CORRECTED: Two-column layout
  mutate_dialog <- rk.XML.dialog(label="Mutate Strings", child=rk.XML.row(
      mutate_selector,
      rk.XML.col(
          mutate_slot,
          rk.XML.frame(mutate_dropdown, label="Mutation Function"),
          mutate_save_obj,
          mutate_preview
      )
  ))

  mutate_component <- rk.plugin.component(
      "Mutate Strings", xml=list(dialog=mutate_dialog),
      js=list(
        require="stringr",
        calculate="
          var string_vec = getValue('mutate_string');
          var op = getValue('mutate_func');
          echo('stringr.mutated <- ' + op + '(string=' + string_vec + ')\\n');
        ",
        preview="
          var string_vec = getValue('mutate_string');
          var op = getValue('mutate_func');
          echo('preview_data <- as.data.frame(' + op + '(string=' + string_vec + '))\\n');
        ",
        printout="
          var save_name = getValue('mutate_save.objectname');
          if(getValue('mutate_save.active')){
            echo('rk.header(\"Mutated string saved as: ' + save_name + '\", level=3)\\n');
          }
        "
      ),
      hierarchy=list("data", "String Manipulation (stringr)")
  )

  # =========================================================================================
  # Component 2: Extract from Strings
  # =========================================================================================
  extract_selector <- rk.XML.varselector(id.name="extract_selector")
  extract_slot <- rk.XML.varslot("String or vector to extract from", source="extract_selector", required=TRUE, id.name="extract_string")
  extract_pattern <- rk.XML.input("Pattern (for extract/match)", id.name="extract_pattern")
  extract_start <- rk.XML.spinbox("Start position (for str_sub)", initial=1, id.name="extract_start")
  extract_end <- rk.XML.spinbox("End position (for str_sub)", initial=5, id.name="extract_end")

  extract_dropdown <- rk.XML.dropdown(label="Function", options=list(
    "Extract by position (str_sub)"=list(val="str_sub", chk=TRUE),
    "Extract first matching group (str_extract)"=list(val="str_extract"),
    "Extract all matching groups (str_extract_all)"=list(val="str_extract_all"),
    "Extract capture groups (str_match)"=list(val="str_match")
  ), id.name="extract_func")

  extract_save_obj <- rk.XML.saveobj("Save result as", initial="stringr.extracted", chk=TRUE, id.name="extract_save")
  extract_preview <- rk.XML.preview(mode = "data")

  # CORRECTED: Two-column layout
  extract_dialog <- rk.XML.dialog(label="Extract from Strings", child=rk.XML.row(
      extract_selector,
      rk.XML.col(
          extract_slot,
          rk.XML.frame(extract_dropdown, extract_pattern, rk.XML.row(extract_start, extract_end), label="Extraction Options"),
          extract_save_obj,
          extract_preview
      )
  ))

  extract_component <- rk.plugin.component(
      "Extract from Strings", xml=list(dialog=extract_dialog),
      js=list(
        require="stringr",
        calculate="
          var string_vec = getValue('extract_string');
          var op = getValue('extract_func');
          var command = 'stringr.extracted <- ' + op + '(string=' + string_vec;
          if (op == 'str_sub') {
            command += ', start=' + getValue('extract_start') + ', end=' + getValue('extract_end');
          } else {
            command += ', pattern=\"' + getValue('extract_pattern') + '\"';
          }
          command += ')';
          echo(command + '\\n');
        ",
        preview="
          var string_vec = getValue('extract_string');
          var op = getValue('extract_func');
          var command = 'preview_data <- as.data.frame(' + op + '(string=' + string_vec;
          if (op == 'str_sub') {
            command += ', start=' + getValue('extract_start') + ', end=' + getValue('extract_end');
          } else {
            command += ', pattern=\"' + getValue('extract_pattern') + '\"';
          }
          command += '))';
          echo(command + '\\n');
        ",
        printout="
          var save_name = getValue('extract_save.objectname');
          if(getValue('extract_save.active')){
            echo('rk.header(\"Extraction result saved as: ' + save_name + '\", level=3)\\n');
          }
        "
      ),
      hierarchy=list("data", "String Manipulation (stringr)")
  )

  # =========================================================================================
  # Component 3: Split Strings
  # =========================================================================================
  split_selector <- rk.XML.varselector(id.name="split_selector")
  split_input <- rk.XML.varslot("String or vector to split", source="split_selector", required=TRUE, id.name="split_string")
  split_pattern <- rk.XML.input("Pattern to split by", id.name="split_pattern")
  split_n_pieces <- rk.XML.spinbox("Number of pieces (for str_split_fixed)", initial=2, id.name="split_n_pieces")

  split_dropdown <- rk.XML.dropdown(label="Function", options=list(
    "Split into a list of pieces"=list(val="str_split", chk=TRUE),
    "Split into a fixed number of pieces"=list(val="str_split_fixed")
  ), id.name="split_func")

  split_save_obj <- rk.XML.saveobj("Save result as", initial="stringr.split", chk=TRUE, id.name="split_save")
  split_preview <- rk.XML.preview(mode = "data")

  # CORRECTED: Two-column layout
  split_dialog <- rk.XML.dialog(label="Split Strings", child=rk.XML.row(
      split_selector,
      rk.XML.col(
          split_input,
          rk.XML.frame(split_dropdown, split_pattern, split_n_pieces, label="Function Options"),
          split_save_obj,
          split_preview
      )
  ))

  split_component <- rk.plugin.component(
      "Split Strings", xml=list(dialog=split_dialog),
      js=list(
        require="stringr",
        calculate="
          var string_vec = getValue('split_string');
          var op = getValue('split_func');
          var command = 'stringr.split <- ' + op + '(string=' + string_vec + ', pattern=\"' + getValue('split_pattern') + '\"';
          if (op == 'str_split_fixed') {
            command += ', n=' + getValue('split_n_pieces');
          }
          command += ')';
          echo(command + '\\n');
        ",
        preview="
          var string_vec = getValue('split_string');
          var op = getValue('split_func');
          var command = 'preview_data <- as.data.frame(' + op + '(string=' + string_vec + ', pattern=\"' + getValue('split_pattern') + '\"';
          if (op == 'str_split_fixed') {
            command += ', n=' + getValue('split_n_pieces');
          }
          command += '))';
          echo(command + '\\n');
        ",
        printout="
          var save_name = getValue('split_save.objectname');
          if(getValue('split_save.active')){
            echo('rk.header(\"Split result saved as: ' + save_name + '\", level=3)\\n');
          }
        "
      ),
      hierarchy=list("data", "String Manipulation (stringr)")
  )

  # =========================================================================================
  # Component 4: Manage Length
  # =========================================================================================
  len_selector <- rk.XML.varselector(id.name="len_selector")
  len_slot <- rk.XML.varslot("String or vector to manage", source="len_selector", required=TRUE, id.name="len_string")
  len_width <- rk.XML.spinbox("Width", initial=10, id.name="len_width")
  len_side_pad <- rk.XML.dropdown("Side", options=list("left"=list(val="left",chk=T), "right"=list(val="right"), "both"=list(val="both")), id.name="len_side")

  len_dropdown <- rk.XML.dropdown(label="Function", options=list(
    "Trim whitespace"=list(val="str_trim", chk=TRUE),
    "Squish whitespace"=list(val="str_squish"),
    "Get string length(s)"=list(val="str_length"),
    "Pad string to a fixed width"=list(val="str_pad"),
    "Truncate string to a fixed width"=list(val="str_trunc")
  ), id.name="len_func")

  len_save_obj <- rk.XML.saveobj("Save result as", initial="stringr.length_managed", chk=TRUE, id.name="len_save")
  len_preview <- rk.XML.preview(mode = "data")

  # CORRECTED: Two-column layout
  len_dialog <- rk.XML.dialog(label="Manage String Length", child=rk.XML.row(
      len_selector,
      rk.XML.col(
        len_slot,
        rk.XML.frame(len_dropdown, len_width, len_side_pad, label="Management Options"),
        len_save_obj,
        len_preview
      )
  ))

  len_component <- rk.plugin.component(
      "Manage String Length", xml=list(dialog=len_dialog),
      js=list(
        require="stringr",
        calculate="
          var string_vec = getValue('len_string');
          var op = getValue('len_func');
          var command = 'stringr.length_managed <- ' + op + '(string=' + string_vec;
          if (op == 'str_pad' || op == 'str_trunc') {
            command += ', width=' + getValue('len_width');
            if (op == 'str_pad') {
               command += ', side=\"' + getValue('len_side') + '\"';
            }
          }
          command += ')';
          echo(command + '\\n');
        ",
        preview="
          var string_vec = getValue('len_string');
          var op = getValue('len_func');
          var command = 'preview_data <- as.data.frame(' + op + '(string=' + string_vec;
          if (op == 'str_pad' || op == 'str_trunc') {
            command += ', width=' + getValue('len_width');
            if (op == 'str_pad') {
               command += ', side=\"' + getValue('len_side') + '\"';
            }
          }
          command += '))';
          echo(command + '\\n');
        ",
        printout="
          var save_name = getValue('len_save.objectname');
          if(getValue('len_save.active')){
            echo('rk.header(\"Result saved as: ' + save_name + '\", level=3)\\n');
          }
        "
      ),
      hierarchy=list("data", "String Manipulation (stringr)")
  )

  # =========================================================================================
  # Final Plugin Skeleton Call
  # =========================================================================================
  rk.plugin.skeleton(
    about = package_about,
    dependencies = dependencies.info,
    xml = list(dialog = main_dialog),
    js = list(
        require = "stringr",
        calculate = js_calc_main,
        preview = js_preview_main,
        printout = js_print_main
    ),
    rkh = list(help = help_main),
    components = list(mutate_component, extract_component, split_component, len_component),
    pluginmap = list(
        name = "Detect & Locate Matches",
        hierarchy = list("data","String Manipulation (stringr)")
    ),
    path = "./rk.stringr",
    create = c("pmap", "xml", "js", "desc", "rkh"),
    load = TRUE,
    overwrite = TRUE,
    show = FALSE
  )

  cat("\nPlugin package 'rk.stringr' with 5 plugins generated under the 'Data' menu.\n")
})
